/**MIT License

Copyright (c) 2016 Daniel Knox

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.**/

#include "mbed.h"
#include "RN2483.h"

Serial pc(USBTX, USBRX);

// Lora Declarations - using unpopulated header on K64F
RN2483 lorabee(D1, D0); // tx, rx
DigitalInOut loraResetPin(PTB9);

// Provisioned on network server.
const uint8_t devEUI[8] =
{0x09, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01};

const uint8_t appEUI[8] =
{0x09, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01};

const uint8_t appKey[16] =
{0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x91, 0x23};

const uint8_t devAddr[4] =
{0x094, 0x19, 0x27, 0xf4};

const uint8_t appSKey[16] =
{0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x91, 0x23};

const uint8_t nwkSKey[16] =
{0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x91, 0x23};

uint8_t testPayload[] =
{
    0x30, 0x31, 0xFF, 0xDE, 0xAD
};

// Foward Declarations
void printMenu();
void hardwareResetLoRa();
int timedRead(Serial &, unsigned long);
size_t readBytesUntil(Serial &, char, char *, size_t);
void configJoinNetworkOTTA();
void configJoinNetworkABP();
void joinOTTA();
void joinABP();
void sleepWakeupTest();
void readVDDTest();
void setBatteryTest();
void setLinkCheckIntervalTest();
void forceEnableTest();
void sendACK();
void send();
void printHWEUI();

int main() {
    pc.baud(38400);
    hardwareResetLoRa();
    printMenu();
    while(true){
        char buffer[5];
        int bytesRead = readBytesUntil(pc, '\n', buffer, 5);
        if(bytesRead>0){
            switch (strtol(buffer,NULL,10)){
             case 1:
                configJoinNetworkOTTA();
                break;
            case 2:
                configJoinNetworkABP();
                break;
            case 3:
                joinOTTA();
                break;
            case 4:
                joinABP();
                break;
            case 5:
                sleepWakeupTest();
                break;
            case 6:
                readVDDTest();
                break;
            case 7:
                setBatteryTest();
                break;
            case 8:
                setLinkCheckIntervalTest();
                break;
            case 9:
                forceEnableTest();
                break;
            case 10:
                sendACK();
                break;
            case 11:
                send();
                break;
            case 12:
                printHWEUI();
                break;
            default:
                printf("Invalid Choice");
                break;    
            }
            Thread::wait(3000);   
            printMenu();
        }
    }
}

void printMenu(){
    printf("------------------------\r\n");
    printf("Config Join OTTA   :  1\r\n");
    printf("Config Join ABP    :  2\r\n");
    printf("Join OTTA          :  3\r\n");
    printf("Join ABP           :  4\r\n");
    printf("Sleep Wakeup       :  5\r\n");
    printf("Read VDD           :  6\r\n");
    printf("Set Battery        :  7\r\n");
    printf("Link Check Test    :  8\r\n");
    printf("Force Enable       :  9\r\n");
    printf("Send Ack MSG       :  10\r\n");
    printf("Send NoAck MSG     :  11\r\n");
    printf("Print HWEUI        :  12\r\n");
    printf("------------------------\r\n");
}

void hardwareResetLoRa() {
    loraResetPin.input();
    Thread::wait(500);
    loraResetPin.output() ;
    loraResetPin = 1;
    Thread::wait(500);
}

void configJoinNetworkOTTA(){
    printf("Network (OTA) Join %s\r\n", lorabee.initOTA(devEUI,appEUI,appKey,true) ? "Success": "Failed");
}

void configJoinNetworkABP(){
    printf("Network (ABP) Join %s\r\n", lorabee.initABP(devAddr, appSKey, nwkSKey, true) ? "Success": "Failed");
}

void joinOTTA(){
    printf("Network (OTTA) Join %s\r\n", lorabee.joinOTTA() ? "Success": "Failed");
}

void joinABP(){
    printf("Network (ABP) Join %s\r\n", lorabee.joinABP() ? "Success": "Failed");
}

void sleepWakeupTest(){
    printf("RN2483 Sleeping for 10 seconds.\r\n");
    lorabee.sleep(10000);
    Thread::wait(20000);
    printf("RN2483 Sleeping for 60 seconds.\r\n");
    lorabee.sleep(60000);
    Thread::wait(20000);
    lorabee.wakeUp();   
    printf("RN2483 was told to wake up.\r\n");
}

void readVDDTest(){
        long vdd;
        if (lorabee.getVDD(&vdd)){
            printf("RN2483 voltage in mV %ld.\r\n", vdd);
        } else {
            printf("Error reading VDD");   
        }
        
}

void setBatteryTest(){
    printf("Battery Setting to 50percent %s\r\n", lorabee.setBattery(125) ? "Success": "Failed");
    Thread::wait(2000);
    printf("Battery Setting to external power %s\r\n", lorabee.setBattery(0) ? "Success": "Failed");
    Thread::wait(2000);
    printf("Battery Setting to unable to read power %s\r\n", lorabee.setBattery(255) ? "Success": "Failed");
}

void setLinkCheckIntervalTest(){
    printf("Set Link Check 120 seconds %s\r\n", lorabee.setLinkCheckInterval(120) ? "Success": "Failed");
    Thread::wait(2000);
    printf("Set Link Check Disable seconds %s\r\n", lorabee.setLinkCheckInterval(0) ? "Success": "Failed");
}

void forceEnableTest(){
    printf("Remove silence %s\r\n", lorabee.forceEnable() ? "Success": "Failed");
}

void printHWEUI(){
    uint8_t buffer[10];
    uint8_t bufferSize = lorabee.getHWEUI(buffer,10);
    printf("Hardware EUI: ");
    if(bufferSize > 0){
        for(int i = 0; i < bufferSize; i++){
            printf("%x",buffer[i]);
        }
    }
    printf("\r\n");
}

void sendACK(){
    switch (lorabee.sendReqAck(1, testPayload, 5, 3))
        {
        case NoError:
            printf("Successful transmission.\r\n");
            break;
        case NoResponse:
            printf("There was no response from the device.\r\n");
            break;
        case Silent:
            printf("The device was silenced\r\n");
            break;
        case Timedout:
            printf("Connection timed-out. Check your serial connection to the device!\r\n");
            break;
        case PayloadSizeError:
            printf("The size of the payload is greater than allowed. Transmission failed!\r\n");
            break;
        case InternalError:
            printf("Serious Error\r\n.");
            break;
        case Busy:
            printf("The device is busy.\r\n");
            break;
        case NetworkFatalError:
            printf("There is a non-recoverable error with the network connection. You should re-connect.\r\n");
            break;
        case NotConnected:
            printf("The device is not connected to the network. Please connect to the network before attempting to send data.\r\n");
            break;
        case NoAcknowledgment:
            printf("There was no acknowledgment sent back!\r\n");
            break;
        default:
            break;
        }
}

void send(){
    switch (lorabee.send(1, testPayload, 5))
        {
        case NoError:
            printf("Successful transmission.\r\n");
            break;
        case NoResponse:
            printf("There was no response from the device.\r\n");
            break;
        case Silent:
            printf("The device was silenced\r\n");
            break;
        case Timedout:
            printf("Connection timed-out. Check your serial connection to the device!\r\n");
            break;
        case PayloadSizeError:
            printf("The size of the payload is greater than allowed. Transmission failed!\r\n");
            break;
        case InternalError:
            printf("Serious Error\r\n.");
            break;
        case Busy:
            printf("The device is busy.\r\n");
            break;
        case NetworkFatalError:
            printf("There is a non-recoverable error with the network connection. You should re-connect.\r\n");
            break;
        case NotConnected:
            printf("The device is not connected to the network. Please connect to the network before attempting to send data.\r\n");
            break;
        default:
            break;
        }  
}

int timedRead(Serial &serialInterface, unsigned long _timeout){
    int c;
    Timer t;
    t.start();
    unsigned long _startMillis = t.read_ms(); // get milliseconds
    while (! serialInterface.readable() && t.read_ms() - _startMillis <_timeout) {
       Thread::yield();
    }
    t.stop();
    if(serialInterface.readable()){
        c = serialInterface.getc();
        if (c >= 0){
            return c;
        }    
    }    
    return -1; // -1 indicates timeout
} 

size_t readBytesUntil(Serial &serialInterface, char terminator, char *buffer, size_t length)
{
    if (length < 1) return 0;
    size_t index = 0;
    while (index < length) {
        int c = timedRead(serialInterface, 1000);
        if (c < 0 || c == terminator) break;
        *buffer++ = (char)c;
        index++;
    }
    return index; // return number of characters, not including null terminator        
} 