#include "mbed.h"


/*
 * Based on 
 *
 * http://mbed.org/users/AjK/notebook/regarding-interrupts-use-and-blocking/?c=9228
 */
 
 
/*

Specification:

    led1: flash briefly once per second, EVERY second.
    led2: On at t = 2 and off at t = 5
    led3: On at t = 3 and off at t = 6
*/

#define SIX

#ifdef ONE
/*
 * Basic processing looping  
 *
 * Good     - simple to write/debug
 * Bad      - not so simple to add led2/led3 without some state machine logic and timer polling
 */
 
 
DigitalOut led1(LED1);
 
int main() {
    while(1) {
        led1 = 1;
        wait(0.2);
        led1 = 0;
        wait(0.2);
    }
}

#endif

#ifdef TWO

/*
 * Event driven logic and loop together
 *
 * Good     - led2 is getting serviced by callback
 * Bad      - led1 will missing some flashes due to wait() in callback
 *
 */
 
Timeout to1;
DigitalOut led1(LED1);
DigitalOut led2(LED2);
 
void cb1(void) { 
    led2 = 1;
    wait(3);
    led2 = 0;
}
 
int main() {
 
    to1.attach(&cb1, 2);
 
    while(1) {
        led1 = 1;
        wait(0.05);
        led1 = 0;
        wait(0.95);
    }
}

#endif
#ifdef THREE

/*
 * Good     - led1/led2/led3 all getting serviced
 * Bad      - timing off on led3 since that callback has to wait for previous callback to finish 
 *          - timing off on led1 due to both callbacks wait()
 */
 
 
Timeout to1;
Timeout to2;
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
 
void cb1(void) { 
    led2 = 1;
    wait(3);
    led2 = 0;
}
 
void cb2(void) { 
    led3 = 1;
    wait(3);
    led3 = 0;
}
 
int main() {
 
    to1.attach(&cb1, 2);
    to2.attach(&cb2, 3);
 
    while(1) {
        led1 = 1;
        wait(0.05);
        led1 = 0;
        wait(0.95);
    }
}
#endif

#ifdef FOUR

/*
 * Good - Use gloabls to coordinate callback/main loop - no wait() in callback
 * Bad  - led1 uses wait() making it less effective for polling other devices
 */
 
 
Timeout to1;
Timeout to2;
DigitalOut led1(LED1);
DigitalOut to1_led(LED2);
DigitalOut to2_led(LED3);
 
bool to1triggered = false;
bool to2triggered = false;
 
void cb1(void) { 
    to1triggered = true;
}
 
void to1handle(void) {
    if (to1_led == 0) {
        to1_led = 1;
        to1.detach();
        to1.attach(&cb1, 3);
    }
    else {
        to1_led = 0;
        to1.detach();            
    } 
}

 
void cb2(void) { 
    to2triggered = true;    
}
 
void to2handle(void) {
    if (to2_led == 0) {
        to2_led = 1;
        to2.detach();
        to2.attach(&cb2, 3);
    }
    else {
        to2_led = 0;
        to2.detach();            
    } 
}
 
 
int main() {
 
    to1.attach(&cb1, 2);
    to2.attach(&cb2, 3);
 
    while(1) {
    
        if (to1triggered) {
            to1triggered = false;
            to1handle();
        }
    
        if (to2triggered) {
            to2triggered = false;
            to2handle();
        }
    
        led1 = 1;
        wait(0.05);
        led1 = 0;
        wait(0.95);
    }
}

#endif

#ifdef FIVE

/*
 * Good - all led's are callback interrupt driven
 * Bad  - printf() in callback (interrupt context) bad idea
 */

#include "mbed.h"
 
Ticker  tled1on;
Timeout tled1off;
Timeout to1;
Timeout to2;
DigitalOut led1(LED1);
DigitalOut to1_led(LED2);
DigitalOut to2_led(LED3);
 
void cb1(void) { 
    if (to1_led == 0) {
        to1_led = 1;
    }
    else {
        to1_led = 0;
    }
    // Reschedule a new event.
    printf("cb1 callback\n");
    to1.detach();
    to1.attach(&cb1, 3);
}
 
void cb2(void) {
    if (to2_led == 0) {
        to2_led = 1;
    }
    else {
        to2_led = 0;
    }
    // Reschedule a new event.
    to2.detach();
    to2.attach(&cb2, 3); 
}
 
void tled1off_cb(void) {
    led1 = 0;
}
 
void tled1on_cb(void) {
    led1 = 1;
    tled1off.detach();
    tled1off.attach(&tled1off_cb, 0.05);
}
 
int main() {
 
    led1 = 1;
    tled1off.attach(&tled1on_cb, 0.05);
    tled1on.attach(&tled1on_cb, 1);
    to1.attach(&cb1, 2);
    to2.attach(&cb2, 3);
 
    while(1) {
        // Calculate PI here as we have so much time 
    }
}

#endif

#ifdef SIX

/*
 * Final version ... event driven for LED's
*/
 
Ticker  tled1on;
Timeout tled1off;
Timeout to1;
Timeout to2;
DigitalOut led1(LED1);
DigitalOut to1_led(LED2);
DigitalOut to2_led(LED3);
 
void cb1(void) { 
    if (to1_led == 0) {
        to1_led = 1;
    }
    else {
        to1_led = 0;
    }
    
    
    // Reschedule a new event.
    to1.detach();
    to1.attach(&cb1, 3);
}
 
void cb2(void) {
    if (to2_led == 0) {
        to2_led = 1;
    }
    else {
        to2_led = 0;
    }
    // Reschedule a new event.
    to2.detach();
    to2.attach(&cb2, 3); 
}
 
void tled1off_cb(void) {
    led1 = 0;
}
 
void tled1on_cb(void) {
    led1 = 1;
    tled1off.detach();
    tled1off.attach(&tled1off_cb, 0.05);
}
 
int main() {
 
    led1 = 1;
    tled1off.attach(&tled1on_cb, 0.05);
    tled1on.attach(&tled1on_cb, 1);
    to1.attach(&cb1, 2);
    to2.attach(&cb2, 3);
 
    while(1) {
        // Calculate PI here as we have so much time :)
        printf("main loop\n");
    }
}

#endif

