/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "gnss.h"
#include "battery_charger_bq24295.h"
#include "onboard_modem_api.h"
#include <SDBlockDevice.h>
#include <FATFileSystem.h>

#include <string>
#include <iostream>

using namespace std;
using std::string;

#define CHECK_TALKER(s) ((buffer[3] == s[0]) && (buffer[4] == s[1]) && (buffer[5] == s[2]))

// Set the minimum input voltage limit for the BQ24295 to 3.8 Volt
#define MIN_INPUT_VOLTAGE_LIMIT_MV  3880

SDBlockDevice sd(PE_6, PE_5, PE_2, PE_11);
FATFileSystem fs("sd");

// User LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

//GNSS 1V8_MAX IO power
DigitalOut GNSSOn(GNSSEN, 1);

// Ethernet socket LED 
DigitalOut ledYellow(LED4,1);

// User Button
#ifdef TARGET_UBLOX_C027
    // No user button on C027
    InterruptIn userButton(NC);
#else
    InterruptIn userButton(SW0);
#endif

// GNSS
GnssSerial gnss;

// i2c3 Bus
I2C i2c3(I2C_SDA_B, I2C_SCL_B);
    
// Battery Charger BQ24295
BatteryChargerBq24295 charger;

// Delay between LED changes in second
volatile float delay = 0.5;

// To check if the user pressed the User Button or not
void threadBodyUserButtonCheck(void const *args){
    while (1){
        if (userButton.read() == 1 ) {
        // User Button is pressed 
            delay = 0.1;
            //Indicate the button is pressed 
            ledYellow = 0;
        }
        else { 
        // User button is released
            delay = 0.5;
            //Turn off the Yellow LED on Ethernet socket
            ledYellow = 1;
        }
    }
}

void SDCardWriter(int time)///////////////////////////////////////////////////////////////////////////////
{
    printf("\nWaiting for SD Card initialization\n\n");
    mkdir("/sd/Amit", 0777);
    FILE *fp = fopen("/sd/Amit/sdBlockDeviceTest.txt", "a");
    if(fp == NULL) {
        printf("\nCould not open file for write\n\n");
        return  ;
    }
    
    fprintf(fp, "\nGNSS Time: %d", time);
    printf("\nGNSS Time Recorded\n\n");
    fclose(fp);
}////////////////////////////////////////////////////////////////////////////////////////////////////////////

int main()
{
    printf("\nu-blox C030\n");
    
    int count = 30;
    int i = 1;
    
    /////////////////////////////////////////////////////////////////////////////////////SD TEST
    string timestring = "test";
    sd.init();
    fs.mount(&sd);
    SDCardWriter(i);
    //fs.unmount();
    //sd.deinit();
    //////////////////////////////////////////////////////////////////////////////////////SD TEST
    
    GnssSerial gnss;
    int gnssReturnCode;
    int length;
    char buffer[256];

    // GNSS initialisation
    if(gnss.init()) {
        printf("GNSS initialised.\n\r");
    }
    else {
        printf("GNSS initialisation failure.\n\r");
    }
   
    // The battery charger initialisation
    charger.init(&i2c3);   
    charger.setInputVoltageLimit(MIN_INPUT_VOLTAGE_LIMIT_MV); 
    // Disable the battery charger's watchdog, otherwise it resets the battry charger
    charger.setWatchdog(0);
   
    // Initialised the modem
    onboard_modem_init();
    
    // Power up the modem
    onboard_modem_power_up();
    
    // Create threadUserButtonCheck thread
    Thread user_button(osPriorityNormal);
    user_button.start(callback(threadBodyUserButtonCheck, (void *)"User Button Thread"));

    
    //Set GNSS IO On
    GNSSOn = 1;
    
    // Set the LED states
    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;
    
    printf("u-blox C030\n\r");
    
    //Main loop
    while(1) {
        wait(delay);
        //Shift the LED states
        int carry = ledBlue;
        ledBlue = ledRed;
        ledRed = ledGreen;
        ledGreen = carry;
        
        //Amittttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttt
        
        SDCardWriter(i);//////////////////////////////////////////////////////////////////////////
        
        if(i >= count)
        {
            fs.unmount();
            sd.deinit();
        }
        
        i++;
        
        gnssReturnCode = gnss.getMessage(buffer, sizeof(buffer));
        if (gnssReturnCode > 0) {
                ledGreen = 0;
                ledBlue = 1;
                ledRed = 1;
                length = LENGTH(gnssReturnCode);
 
                printf("NMEA: %.*s\n", length - 2, buffer);
 
                if ((PROTOCOL(gnssReturnCode) == GnssParser::NMEA) && (length > 6)) {
                    // Talker is $GA=Galileo $GB=Beidou $GL=Glonass $GN=Combined $GP=GNSS
                    if ((buffer[0] == '$') || buffer[1] == 'G') {
                      if (CHECK_TALKER("GLL")) {
                            double latitude = 0, longitude = 0;
                            char ch;
 
                            if (gnss.getNmeaAngle(1, buffer, length, latitude) && 
                                gnss.getNmeaAngle(3, buffer, length, longitude) && 
                                gnss.getNmeaItem(6, buffer, length, ch) && (ch == 'A')) {
                                ledBlue = 0;
                                ledRed = 0;
                                ledGreen = 0;
 
                                printf("\nGNSS: location is %.5f %.5f.\n\n", latitude, longitude);
                                printf("I am here: https://maps.google.com/?q=%.5f,%.5f\n\n",
                                       latitude, longitude); 
                            }
                        } else if (CHECK_TALKER("GGA") || CHECK_TALKER("GNS")) {
                            double altitude = 0; 
                            const char *timeString = NULL;
 
                            // Altitude
                            if (gnss.getNmeaItem(9, buffer, length, altitude)) {
                                printf("\nGNSS: altitude is %.1f m.\n", altitude); 
                            }
 
                            // Time
                            timeString = gnss.findNmeaItemPos(1, buffer, buffer + length);
                            if (timeString != NULL) {
                                ledBlue = 0;
                                ledRed = 1;
                                ledGreen = 1;
 
                                printf("\nGNSS: time is %.6s.\n\n", timeString);
                                
                            }
                        } else if (CHECK_TALKER("VTG")) {
                            double speed = 0; 
 
                            // Speed
                            if (gnss.getNmeaItem(7, buffer, length, speed)) {
                                printf("\nGNSS: speed is %.1f km/h.\n\n", speed);
                            }
                        }
                    }
                }
            }
        
        
    }
}

// End Of File