/* mbed tmpLM74 Library, for a LM74 based digital thermometer
 * Copyright (c) 2011, atpolitis, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MBED_TMPLM74_H
#define MBED_TMPLM74_H

#include "mbed.h"

#define INVALID_LM74_TEMP   0X961
#define MAX_LM74_TEMP       (INVALID_LM74_TEMP-1)*0.0625

/** An LM74 based digital thermometer
 *
 * Currently supports SPI peripheral interface
 *
 * @code
 * #include "mbed.h"
 * #include "tmpLM74.h"
 * 
 * Serial pc(USBTX, USBRX); // tx, rx
 * DigitalOut myled(LED1);
 *
 * TmpLM74 Temp74(p5, p6, p7, p8);
 *
 * //   mbed connections:
 * //                       mosi(p5) to LM74 pin 1, through 10k
 * //                       miso(p6) to LM74 pin 1
 * //                       sclk(p7) to LM74 pin 2
 * //                       miso(p8) to LM74 pin 7
 *
 * //TmpLM74.startLM74();       // CAN be used, but NOT necessary (invoked within readTemp(), if needed)
 *
 * int main() {
 *
 *      float TempC;
 *
 *      pc.printf("HELLO!,  testing LM74 temperature sensor ...\n\r");
 *
 *      for (int i = 0; i < 10; i++){
 *          wait(2);
 *          TempC = Temp74.readTemp();
 *          pc.printf("Temperature = ");
 *          if(TempC > MAX_LM74_TEMP) {         // check whether temp is valid
 *              pc.printf(" ? *C\n", TempC);
 *          } else {
 *              pc.printf("%3.1f *C\n", TempC);
 *          }
 *          myled = !myled;
 *      }
 *      Temp74.shutLM74down();
 *      while(1){
 *          wait(1);
 *          myled = !myled;
 *      }
 * }
 * @endcode
 */
//  ********************************************************************
//      in this version, class TmpLM74 is derived from class SPI
//      and inherits its members
//
//  ********************************************************************

class TmpLM74: public SPI {

    public:
    
        TmpLM74(PinName mosi, PinName miso, PinName sclk, PinName csLM74);
        
        float readTemp(void);
        void startLM74(void);
        void shutLM74down(void);
        
    protected:
        DigitalOut _csLM74;
};

#endif