/* 
 * Copyright (c) 2015 Robert Walker
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __BLE_CYCLING_SPEED_AND_CADENCE_SERVICE_H__
#define __BLE_CYCLING_SPEED_AND_CADENCE_SERVICE_H__

#include "ble/BLE.h"

/**
* @class CyclingSpeedAndCadenceService
* @brief BLE Service for Cycling Speed and Cadence. This BLE Service contains the location of the sensor, the total wheel revolutions, total crank revolutiosn. <br>
* Service:  https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.cycling_speed_and_cadence.xml <br>
* CSC Char: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.csc_measurement.xml <br>
* Location: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.sensor_location.xml
*/
class CyclingSpeedAndCadenceService {
public:
    static const uint8_t FLAG_WHEEL_PRESENT = (1 << 0);
    static const uint8_t FLAG_CRANK_PRESENT = (1 << 1);

    enum Mode {
        MODE_SPEED = FLAG_WHEEL_PRESENT,
        MODE_CADENCE = FLAG_CRANK_PRESENT,
        MODE_SPEED_CADENCE = FLAG_WHEEL_PRESENT | FLAG_CRANK_PRESENT
    };
    
    /**
    * @enum SensorLocation
    * @brief Location of sensor on bike.
    */
    enum {
        LOCATION_OTHER,        /*!< Other */
        LOCATION_TOP_OF_SHOE,  /*!< Top of shoe */
        LOCATION_IN_SHOE,      /*!< In shoe */
        LOCATION_HIP,          /*!< Hip */
        LOCATION_FRONT_WHEEL,  /*!< Front Wheel */
        LOCATION_LEFT_CRANK,   /*!< Left Crank */
        LOCATION_RIGHT_CRANK,  /*!< Right Crank */
        LOCATION_LEFT_PEDAL,   /*!< Left Pedal */
        LOCATION_RIGHT_PEDAL,  /*!< Right Pedal */
        LOCATION_FRONT_HUB,    /*!< Front Hub */
        LOCATION_REAR_DROPOUT, /*!< Rear Dropout */
        LOCATION_CHAINSTAY,    /*!< Chainstay */
        LOCATION_REAR_WHEEL,   /*!< Rear Wheel */
        LOCATION_REAR_HUB,     /*!< Rear Hub */
        LOCATION_CHEST,        /*!< Chest */
    };
    
    enum {
        UUID_SENSOR_LOCATION_CHAR = 0x2A5D,
        UUID_SC_CONTROL_POINT_CHAR = 0x2A55
    };

public:
    /**
     * @brief Constructor with initial counter values.
     *
     * @param[ref] _ble
     *               Reference to the underlying BLE.
     * @param[in] wheelCounter (32-bit)
     *               initial value for the wheel counter.
     * @param[in] crankCounter (32-bit)
     *               initial value for the crank counter.
     * @param[in] location
     *               Sensor's location.
     */
    CyclingSpeedAndCadenceService(BLE &_ble, Mode _mode, uint8_t location) :
        ble(_ble),
        value(_mode),     
        mode(_mode),
        csc(GattCharacteristic::UUID_CSC_MEASUREMENT_CHAR,
            value.bytes,
            value.getSize(), value.getSize(),
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        cscFeat(GattCharacteristic::UUID_CSC_FEATURE_CHAR, (uint8_t*)&mode,
            2, 2,
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        scLocation(UUID_SENSOR_LOCATION_CHAR, &location),
        controlPoint(UUID_SC_CONTROL_POINT_CHAR, &controlPointValue) {
        setupService();
    }

    /**
     * @brief Set a new value for wheel revolutions.
     *
     * @param[in] wheelCounter
     *                  Total wheel revolutions.
     * @param[in] eventTime
     *                  Time of event.
     */
    void updateWheelCounter(uint32_t wheelCounter, uint16_t eventTime)
    {
        value.updateWheelCounter(wheelCounter, eventTime);
        sendUpdate();
    }

    /**
     * @brief Set a new value for crank revolutions.
     *
     * @param[in] crankCounter
     *                  Total crank revolutions.
     * @param[in] eventTime
     *                  Time of event.
     */
    void updateCrankCounter(uint16_t crankCounter, uint16_t eventTime)
    {
        value.updateCrankCounter(crankCounter, eventTime);
        sendUpdate();
    }

    void updateCounters(uint32_t wheelCounter, uint16_t crankCounter, uint16_t eventTime)
    {
        value.updateCounters(wheelCounter, crankCounter, eventTime);
        sendUpdate();
    }

    /**
     * This callback allows the CyclingSpeedAndCadenceService to receive updates to the
     * controlPoint Characteristic.
     *
     * @param[in] params
     *     Information about the characterisitc being updated.
     */
    virtual void onDataWritten(const GattWriteCallbackParams *params) {
        if (params->handle == controlPoint.getValueAttribute().getHandle()) {
            /* Do something here if the new value is 1; else you can override this method by
             * extending this class.
             * @NOTE: if you are extending this class, be sure to also call
             * ble.onDataWritten(this, &ExtendedHRService::onDataWritten); in
             * your constructor.
             */
        }
    }

protected:
    void setupService(void) {
        GattCharacteristic *charTable[] = {&csc, &cscFeat, &scLocation, &controlPoint};
        GattService         cscService(GattService::UUID_CYCLING_SPEED_AND_CADENCE, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(cscService);
        ble.onDataWritten(this, &CyclingSpeedAndCadenceService::onDataWritten);
    }


    void sendUpdate()
    {            
        ble.gattServer().write(csc.getValueHandle(), value.bytes, value.getSize());
    }

protected:
    static const uint16_t MAX_BYTES = (1 + 4 + 2 + 2 + 2);
    union SpeedCadenceValue
    {
        SpeedCadenceValue(uint16_t mode)
        {
            switch (mode)
            {
            case MODE_SPEED:
                v.flags = FLAG_WHEEL_PRESENT;
                break;
            case MODE_CADENCE:
                v.flags = FLAG_CRANK_PRESENT;
                break;
            case MODE_SPEED_CADENCE:
                v.flags = FLAG_WHEEL_PRESENT | FLAG_CRANK_PRESENT;
                break;
            default:
                v.flags = 0;
                break;
            }
        }
        
        void updateWheelCounter(uint32_t wheelCounter, uint16_t eventTime)
        {
            switch(v.flags)
            {
            case FLAG_WHEEL_PRESENT:
                v.v.speed.wheelCounter = wheelCounter;
                v.v.speed.lastWheelEvent = eventTime;
                break;
            case FLAG_WHEEL_PRESENT | FLAG_CRANK_PRESENT:
                v.v.speedCadence.wheelCounter = wheelCounter;
                v.v.speedCadence.lastWheelEvent = eventTime;
                break;
            default:
                break;
            }
        }

        void updateCrankCounter(uint16_t crankCounter, uint16_t eventTime)
        {
            switch(v.flags)
            {
            case FLAG_CRANK_PRESENT:
                v.v.cadence.crankCounter = crankCounter;
                v.v.cadence.lastCrankEvent = eventTime;
                break;
            case FLAG_WHEEL_PRESENT | FLAG_CRANK_PRESENT:
                v.v.speedCadence.crankCounter = crankCounter;
                v.v.speedCadence.lastCrankEvent = eventTime;
                break;
            default:
                break;
            }
        }

        void updateCounters(uint32_t wheelCounter, uint16_t crankCounter, uint16_t eventTime)
        {
            switch(v.flags)
            {
            case FLAG_WHEEL_PRESENT:
                v.v.speed.wheelCounter = wheelCounter;
                v.v.speed.lastWheelEvent = eventTime;
                break;
            case FLAG_CRANK_PRESENT:
                v.v.cadence.crankCounter = crankCounter;
                v.v.cadence.lastCrankEvent = eventTime;
                break;
            case FLAG_WHEEL_PRESENT | FLAG_CRANK_PRESENT:
                v.v.speedCadence.wheelCounter = wheelCounter;
                v.v.speedCadence.lastWheelEvent = eventTime;
                v.v.speedCadence.crankCounter = crankCounter;
                v.v.speedCadence.lastCrankEvent = eventTime;
                break;
            default:
                break;
            }
        }

        uint16_t getSize()
        {
            return 1 +
                ((v.flags & FLAG_WHEEL_PRESENT) ? (4+2) : 0) +
                ((v.flags & FLAG_CRANK_PRESENT) ? (2+2) : 0);
        }
        
        __packed struct Value
        {        
            uint8_t flags;
            __packed union
            {
                __packed struct Speed
                {
                    uint32_t wheelCounter;
                    uint16_t lastWheelEvent;
                } speed;
                __packed struct Cadence
                {
                    uint16_t crankCounter;
                    uint16_t lastCrankEvent;
                } cadence;
                __packed struct SpeedCadence
                {
                    uint32_t wheelCounter;
                    uint16_t lastWheelEvent;
                    uint16_t crankCounter;
                    uint16_t lastCrankEvent;
                } speedCadence;
            } v;
        } v;
        uint8_t bytes[1+4+2+2+2];
    };

protected:
    BLE                 &ble;

    SpeedCadenceValue    value;
    uint16_t             mode;
    uint8_t              controlPointValue;

    GattCharacteristic                   csc;
    GattCharacteristic                   cscFeat;
    ReadOnlyGattCharacteristic<uint8_t>  scLocation;
    WriteOnlyGattCharacteristic<uint8_t> controlPoint;
};

#endif /* #ifndef __BLE_CYCLING_SPEED_AND_CADENCE_SERVICE_H__*/