/*
Copyright (c) 2010 Peter Barrett

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "Utils.h"
#include "Socket.h"

#define MAX_SOCKET_HANDLERS 3
#define MAX_SOCKETS 16

class SocketInternalPad
{
    public:
    SocketInternal si;
    u8  pad[8];
};

class SocketManager
{
    SocketHandler*  _handlers[MAX_SOCKET_HANDLERS];
    SocketInternalPad  _sockets[MAX_SOCKETS];

    public:
    SocketManager()
    {
        memset(_handlers,0,sizeof(_handlers));
        memset(_sockets,0,sizeof(_sockets));
    }

    SocketHandler* GetHandler(int type)
    {
        if (type < 1 || type > MAX_SOCKET_HANDLERS)
            return 0;
        return _handlers[type - 1];
    }

    SocketInternal* GetInternal(int s)
    {
        if (s < 1 || s > MAX_SOCKETS)
            return 0;
        return &_sockets[s - 1].si;
    }

    int RegisterSocketHandler(int type, SocketHandler* handler)
    {
       if (type < 1 || type > MAX_SOCKET_HANDLERS)
            return ERR_SOCKET_TYPE_NOT_FOUND;
        _handlers[type - 1] = handler;
        return 0;
    }

    int Open(int type, SocketAddrHdr* addr, SocketCallback callback, void* userData)
    {
        SocketHandler* h = GetHandler(type);
        if (!h)
            return ERR_SOCKET_TYPE_NOT_FOUND;
        
        for (int i = 0; i < MAX_SOCKETS; i++)
        {
            SocketInternal* si = (SocketInternal*)(_sockets+i);
            if (si->ID == 0)
            {
                si->ID = i+1;
                si->Type = type;
                si->Callback = callback;
                si->userData = userData;
                return h->Open(si,addr);
            }
        }
        return ERR_SOCKET_NONE_LEFT;
    }

    int Send(int socket, const u8* data, int len)
    {
        SocketInternal* si = GetInternal(socket);
        if (!si || si->ID != socket)
            return ERR_SOCKET_NOT_FOUND;
        return GetHandler(si->Type)->Send(si,data,len);
    }

    int Close(int socket)
    {
        SocketInternal* si = GetInternal(socket);
        if (!si || si->ID != socket)
            return ERR_SOCKET_NOT_FOUND;
        si->ID = 0;
        return GetHandler(si->Type)->Close(si);
    }
};

SocketManager gSocketManager;

int Socket_Open(int type, SocketAddrHdr* addr, SocketCallback callback, void* userData)
{
    return gSocketManager.Open(type,addr,callback,userData);
}

int Socket_Send(int socket, const u8* data, int len)
{
    return gSocketManager.Send(socket,data,len);
}

int Socket_Close(int socket)
{
    return gSocketManager.Close(socket);
}

int RegisterSocketHandler(int type, SocketHandler* handler)
{
    return gSocketManager.RegisterSocketHandler(type,handler);
}

SocketInternal* GetSocketInternal(int socket)
{
    return gSocketManager.GetInternal(socket);
}

