/* Console.h */
/* Copyright (C) 2012 Ashley Mills, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
#pragma once
#include "mbed.h"

#define BUFFER_WAIT_TIMEOUT_MS 2000

/** Console interface with line editing and command history.
 */
class Console {
               
    public:
        /** Construct an interface to a serial terminal.
         * 
         * @param terminal The serial device to interface to.
         * @param lineLength The maximum line-length to process.
         * @param numLines The number of lines to hold in the command-history.
         * @param characterBased true for character based serial devices, false for line based.
         */
        Console(Serial *terminal, int lineLength, int numLines, bool characterBased);
        ~Console();
        /** This is called by the instantiating program to update the console.
         */
        virtual void update();
    
    protected:
        char *_lineBuffer;             //< Buffer for current-line.
        int _lineLength;               //< Length of current-line buffer.
        int _numLines;                 //< Number of lines in the command history.
        Serial *_terminal;             //< Pointer to terminal interface.
        /// Whether or not this console is character based (true) or line based (false).
        bool _characterBased;
        
    protected:
        void storeBuffer();            //< Store the current-line in the command history.
        void clearBuffer();            //< Clears the current-line buffer.
        
    private:
        char **_lines;                 //< Command history
        int _currentLine;              //< Index of current line in command history.
        int _usedLines;                //< Number of lines that have been used in the command history.
        
        long _bufferWaitTimeout_ms;    //< This is used when multi-byte sequences are predicted.
      
        int _linePos;                  //< Keeps track of the cursor in the current-line.
        int _prevLinePos;              //< Used when a new line is selected from command history, to clear old line.

        bool waitForInput();           //< Waits for _bufferWaitTimeout_ms for input from the console.
        void loadPreviousBuffer();     //< Load the previous line in the command history.
        void loadNextBuffer();         //< Load the next line in the command history.
        
        void printLineBuffer();        //< Prints the current-line buffer.
        void printHistory();           //< Print the entire command-history.
        void printBuffer(int index);   //< Print buffer at index in the command-history.
        void reprintBuffer();          //< Clears old buffer and reprints it, used for insertion etc.
        
        void updateCharacterBased();   //< Called by update() for character-based consoles.
        void updateLineBased();        //< Called by update() for line-based consoles.
    
        virtual void processCommand(); //< Process a command retrieved from the console.
          
};