#ifndef HPP__UTILS_BUFFER
#define HPP__UTILS_BUFFER

#include "mbed.h"
// #include <cstdlib>

/// @namespace Utils
/// Espace de nom reservé au outils et algorithmes.
namespace Utils {
    /// @namespace Utils::Buffer
    /// Espace de nom reservé au manipulation de zone de memoire.
    namespace Buffer {

        /// @class Buffer
        /// Cette classe permet de crée une zone de memoire de taille fixe ou tous les elements sont d'un type.
        ///
        /// @b Exemple:
        ///
        /// @code
        /// #include "utils/buffer.hpp"
        ///
        /// int main () {
        ///     Utils::Buffer::Buffer<int> buff(10);
        ///     int sum = 0;
        ///
        ///     for ( int i=0 ; i < buff.length() ; i++ ){
        ///         buff [i] = i*2;
        ///         printf( "buff[%d] = %d\n", i, buff [i] );
        ///     }
        ///
        ///     for ( int i=0 ; i < buff.length() ; i++ )
        ///         sum += buff [i];
        ///
        ///     printf( "\nSomme des elements: %d\n" , sum );
        /// }
        /// @endcode
        template<typename T>
        class Buffer {
        protected:
            size_t _size; //< Taille du buffer.
            T     *_buff; //< Pointeur sur la zone de memoire du buffer.
        public:
            /// Defini un buffer de la taille voulu.
            /// @param size Nombre d'element dans le buffer.
            Buffer( const size_t size )
            : _size(size), _buff(new T[_size]) {}

            /// Copie le buffer voulu.
            /// @param b Le buffer à copier.
            Buffer( Buffer<T>& b )
            : _size(b.length()), _buff(new T[_size]) {
                for(int i=0 ; i<_size ; i++)
                    _buff[i] = b[i];
            }

            /// Renvoit la taille du buffer.
            /// @return Taille du buffer.
            virtual inline size_t length () const {
                return this->_size;
            };

            /// Permet d'acceder à une case memoire du buffer.
            /// @param Indexe de la case memoire à acceder.
            /// @return La case memoire indiquée.
            virtual T& operator [] ( int i ) const {
                return this->_buff[ i % _size ];
            }

            /// Met toutes les valeurs du buffer à une valeur constante.
            /// @param val Valeur à appliquer.
            virtual void set (T val) {
                for(size_t i=0 ; i<_size ; i++)
                    _buff[i] = val;
            }

            /// Applique une fonction à toutes les valeurs du buffer.
            /// @param f Function à appliquer. f a pour prototype : void f ( T& val );
            virtual void map ( void (&f)(T) ) {
                for(size_t i=0 ; i<_size ; i++)
                    f(_buff[i]);
            }

        };

        /// class Circular
        /// Defini un buffer circulaire simple.
        /// Un buffer circulaire est un buffer normal mais ou l'origine peut etre deplacer d'vant en arriere, cela permet d'eviter de decaller toutes les valeurs une a une.
        ///
        /// @b Exemple:
        ///
        /// @code
        /// #include "utils/buffer.hpp"
        ///
        /// int main () {
        ///     Utils::Buffer::Circular buff(10);
        /// }
        /// @endcode
        template<typename T>
        class Circular: public Buffer<T> {
        protected:
            int _pos;
        public:
            Circular ( const size_t size )
            : Buffer<T>(size), _pos(0) {}

            virtual T& operator [] ( int i ) const {
                return this->_buff[ ( this->_size + _pos + i )%this->_size ];
            }

            /// Decalage des valeurs vers la "droite", la valeur n devient la valeur n-i.
            /// @param i valeur de decalage.
            virtual Circular& operator >> (int i) {
                _pos = (this->_size+ _pos +i)%this->_size;
                return *this;
            }

            /// Decalage des valeurs vers la "gauche", la valeur n devient la valeur n+i.
            virtual Circular& operator << (int i) {
                _pos = (this->_size+ _pos -i)%this->_size;
                return *this;
            }
        };

    }
}

#endif
