#include "robot/control/servo.hpp"

using namespace Robot::Control;

//-----------------------------------------------------------------------------
/// Definit un servomoteur sur la broche indiquée.
/// La commande ce fera par defaut à 50Hz et l'entrée devra etre entre 0 et 1.
/// @param pin Broche du servomoteur.
Servo::Servo ( PinName pin )
: _out(pin), _f(50), _min(0), _max(1) {
    _out.period( 1/_f );
}

//-----------------------------------------------------------------------------
/// @param pin Broche du servomoteur.
/// @param f   Frequence de commande du servomoteur (Generalement 50 ou 100Hz).
Servo::Servo (PinName pin, float f)
: _out(pin), _f(f), _min(0), _max(1) {
    _out.period( 1/_f );
}

//-----------------------------------------------------------------------------
/// @param pin Broche du servomoteur.
/// @param min Valeur minimum de commande.
/// @param max Valeur maximum de commande.
Servo::Servo (PinName pin, float min, float max)
: _out(pin), _f(50), _min(min), _max(max) {
    _out.period( 1/_f );
}

//-----------------------------------------------------------------------------
/// @param pin Broche du servomoteur.
/// @param f   Frequence de commande du servomoteur (Generalement 50 ou 100Hz).
/// @param min Valeur minimum de commande.
/// @param max Valeur maximum de commande.
Servo::Servo (PinName pin, float f, float min, float max)
: _out(pin), _f(f), _min(min), _max(max) {
    _out.period( 1/_f );
}

//-----------------------------------------------------------------------------
/// Definir la valeur de commande.
/// @param in  Valeur de commande.
void Servo::write (float in) {
    // Envoi des impulsions pouvant varier de 1ms à 2ms.
    _out.pulsewidth ( ( 1 + (in-_min)/(_max-_min) ) / 1000 );
}

//-----------------------------------------------------------------------------
/// Definir la valeur minimum que peut prendre la commande.
/// @param min Valeur minimum de commande.
void Servo::set_min (float min) {
    _min = min;
}

//-----------------------------------------------------------------------------
/// Definir la valeur maximum que peut prendre la commande.
/// @param max Valeur maximum de commande.
void Servo::set_max (float max) {
    _max = max;
}

//-----------------------------------------------------------------------------
/// Definir la valeur maximum que peut prendre la commande.
/// @param f   Frequence de commande du servomoteur (Generalement 50 ou 100Hz).
void Servo::set_frequency (float f) {
    _f = f;
    _out.period( 1/_f );
}
