#include "robot/control/motor.hpp"

using namespace Robot::Control;

/// Constructeur de la classe Motor. Permet de crée un objet de type moteur
/// pour controller un moteur CC.
/// @param in   Mesure de la vitesse du moteur.
/// @param out  Interface de controle du moteur (Exemple : commande de Servo).
/// @param corr Correcteur du systeme.
/// @param f    Frequence (en Hz) d'echantillonnage pour la commande du moteur.
Motor::Motor ( Robot::Sensor::Sensor   &in, 
               Robot::Control::Control &out,
               Utils::System::System   &corr,
               float                          f   )
: _in(in),_out(out),_f(f),_corrector(corr) {
    ticker.attach <Motor> ( this, & Motor::sync_write, 1/_f );
}

//-----------------------------------------------------------------------------
/// Lire la vitesse reélle du moteur.
/// @return Vitesse réelle du moteur à la derniere messure.
float Motor::read () {
    return _mesure;
}

//-----------------------------------------------------------------------------
/// Definir la vitesse de commande du moteur.
/// @param speed Vitesse desirée.
void Motor::write ( float speed ) {
    _speed = speed; // [rad/s]
}

//-----------------------------------------------------------------------------
/// Lecture de la vitesse, calculs du correcteur et ecriture sur la commande.
void Motor::sync_write () {
    _mesure = _in.read() *_f;
    float error = _speed - _mesure, // [rad/s]
          out   = _corrector.step( error );

    _out.write( out );
    // _simu.step( out );
    // compart(_simu, _in, Error);
}

MotorDC::MotorDC ( PinName in1, PinName in2, PinName en, float freq )
: _IN(in1), _DIR(in2), _EN(en), _max(1) {
    _IN.period(1/freq);
    write(0);
}

void MotorDC::write (float in) {
    if ( in == 0 ) {
        _EN = false;
        return;
    }
    _EN = true;
    _DIR =  in < 0;
    _IN.write ( ( _DIR ? 1.0-fabs(in) : fabs(in) ) / _max );
}
