
/*
Copyright (c) 2015 John Magyar

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "QT2100.h"
#include "mbed.h"

/*

Mbed library for the Atmel QT2100 Capacative Touch sensor device.

The QT2100 is a capacitive touch controller device with 10 configurable QTouch channels,
supporting up to seven touch buttons, and either a single slider (or wheel).

Datasheet: http://www.atmel.com/images/at42qt2100_e_level0%2013ww04.pdf

*/

QT2100::QT2100(PinName mosi,
               PinName miso,
               PinName sck,
               PinName cs,
               PinName tx,
               PinName rx):
    _spi(mosi, miso, sck),
    _cs(cs),
    _pc(tx, rx)
{
}

QT2100::~QT2100()
{
}

int qt2100[3];

void QT2100::init()     // SPI interface initialization
{
    _cs = 1;
    _spi.format(8,3);
    _spi.frequency(750000);
    wait_ms(600);
}

void QT2100::xfer3bytes(int byte0, int byte1, int byte2)     //  3-byte SPI transfer
{
    int xfer[3] = {byte0, byte1, byte2};

    for (int i=0; i<3; i++) {
        _cs = 0;
        qt2100[i] = _spi.write(xfer[i]);
        _cs = 1;
        wait_us(100);
    }
}

void QT2100::terminalDisplay()     //  print register values for debugging
{
    _pc.printf("\n");
    for (int i=0; i<3; i++) {
        _pc.printf("QT2100 Byte %d: 0x%x\n",i, qt2100[i]);
    }
}

void QT2100::devId()     //  query Device ID and Firmware version
{
    xfer3bytes(0x40, 0x00, 0x00);     //  flush registers
    xfer3bytes(0x40, 0x00, 0x00);     //  get id and version
    terminalDisplay();
}

void QT2100::verifyChannels()     //  query sensors B1 through B7 and A1 through A3
{
    xfer3bytes(0x20, 0x00, 0x00);     //  flush registers
    for (int i=0; i<10; i++) {
        xfer3bytes(0x20, i, 0x00);     // get each channel's status
        terminalDisplay();
    }
}

int8_t QT2100::keys()     // get data from each key channel
{
    int8_t activeKey;

    xfer3bytes(0x08, 0x78, 0x41);

    switch(qt2100[1]) {
        case (0x01):
            activeKey = 0x01;
            break;
        case (0x02):
            activeKey = 0x02;
            break;
        case (0x04):
            activeKey = 0x04;
            break;
        case (0x08):
            activeKey = 0x08;
            break;
        case (0x10):
            activeKey = 0x10;
            break;
        case (0x20):
            activeKey = 0x20;
            break;
        case (0x40):
            activeKey = 0x40;
            break;
        case (0x80):
            activeKey = 0x80;
            break;
        default:
            activeKey = 0x00;
    }
    return activeKey;
}

int8_t QT2100::slider()     // get data from the slider channels
{
    int8_t sliderPosition;

    xfer3bytes(0x08, 0x78, 0x40);     // 3 bits resolution, 8 positions (expandable to 8 bits, 256 positions)

    switch(qt2100[2]) {
        case (0x0):
            sliderPosition = 0x01;
            break;
        case (0x01):
            sliderPosition = 0x03;
            break;
        case (0x02):
            sliderPosition = 0x07;
            break;
        case (0x03):
            sliderPosition = 0x0f;
            break;
        case (0x04):
            sliderPosition = 0x1f;
            break;
        case (0x05):
            sliderPosition = 0x3f;
            break;
        case (0x06):
            sliderPosition = 0x7f;
            break;
        case (0x07):
            sliderPosition = 0xff;
            break;
        default:
            sliderPosition = 0x00;
    }
    return sliderPosition;
}

int8_t QT2100::wheel()     // get data from the wheel channels
{
    int8_t wheelPosition;

    xfer3bytes(0x00, 0x78, 0x40);     // 3 bits resolution, 8 positions (expandable to 8 bits, 256 positions)

    switch(qt2100[2]) {
        case (0x0):
            wheelPosition = 0x01;
            break;
        case (0x01):
            wheelPosition = 0x03;
            break;
        case (0x02):
            wheelPosition = 0x07;
            break;
        case (0x03):
            wheelPosition = 0x0f;
            break;
        case (0x04):
            wheelPosition = 0x1f;
            break;
        case (0x05):
            wheelPosition = 0x3f;
            break;
        case (0x06):
            wheelPosition = 0x7f;
            break;
        case (0x07):
            wheelPosition = 0xff;
            break;
        default:
            wheelPosition = 0x00;
    }
    return wheelPosition;
}
