
#include "mbed.h"


extern "C" {
#include "pstorage.h"
#include "nrf_error.h"
}

pstorage_handle_t flash_handle;
pstorage_handle_t block_handle;
const uint32_t test_passed_code = 0xC0FFEE;

volatile uint8_t pstorage_wait_flag = 0;
static pstorage_block_t pstorage_wait_handle = 0;

static void flash_cb_handler(pstorage_handle_t  * handle,
                             uint8_t              op_code,
                             uint32_t             result,
                             uint8_t            * p_data,
                             uint32_t             data_len)
{
    if(handle->block_id == pstorage_wait_handle) {
        pstorage_wait_flag = 0;    //If we are waiting for this callback, clear the wait flag.
    }
}

int test_is_passed(void)
{
    pstorage_module_param_t param;
    uint32_t retval;
    uint32_t present;

    retval = pstorage_init();
    if(retval != NRF_SUCCESS) {
        return 0;
    }

    param.block_size  = 32;                 //Select block size of 32 bytes
    param.block_count = 8;                  //Select 8 blocks, total of 256 bytes
    param.cb          = flash_cb_handler;   //Set the pstorage callback handler

    retval = pstorage_register(&param, &flash_handle);
    if (retval != NRF_SUCCESS) {
        return 0;
    }

    //Get block identifiers
    pstorage_block_identifier_get(&flash_handle, 0, &block_handle);

    retval = pstorage_load((uint8_t *)&present, &block_handle, 4, 0);              //Read from flash, only one block is allowed for each pstorage_load command
    if (retval != NRF_SUCCESS) {
        return 0;
    }

    if (present != test_passed_code) {
        return 0;
    }
    
    return 1;
}

int test_save_result(int result)
{
    if (result == 0) {
        uint32_t retval;
        pstorage_wait_handle = block_handle.block_id;            //Specify which pstorage handle to wait for
        pstorage_wait_flag = 1;                                    //Set the wait flag. Cleared in the example_cb_handler
    
        //Store data to three blocks. Wait for the last store operation to finish before reading out the data.
        retval = pstorage_store(&block_handle, (uint8_t *)&test_passed_code, 4, 0);     //Write to flash, only one block is allowed for each pstorage_store command
        if (retval != NRF_SUCCESS) {
            return -1;
        }
    
        // while(pstorage_wait_flag) { }              // until store operation is finished.
    }
    
    return 0;
}


int test_check_io()
{
    // 10x2 pins
    const uint32_t mask = ((uint32_t)1 << 11) - 1;
    BusInOut group1(p18, p17, p23, p24, p0, p2, p3, p4, p9, p10);
    BusInOut group2(p25, p28, p29, p30, p5, p6, p7, p8, p12, p13);

    group1.input();
    group2.output();
    
    
    for (int i = 0; i < 10; i++) {
        uint32_t out = 1 << i;
        group2.write(out);
        if ((group1.read() & mask) != out) {
            return -1;
        }
    }

    group2.input();
    group1.output();

    for (int i = 0; i < 10; i++) {
        uint32_t out = 1 << i;
        group1.write(out);
        if ((group2.read() & mask) != out) {
            return -1;
        }
    }

    group1.input();

    return 0;
}

int test_check_vcc()
{
    AnalogIn vcc(p1);
    float sum = 0;

    for (int i = 0; i < 3; i++) {
        sum += vcc.read();
    }

    float val = ((sum / 3.0) * 3.3);
    if (val < 2.65 || val > 2.95) {   // 2.8V
        return -1;
    }

    return 0;
}

void test_output_result(int result)
{
    const uint32_t periods[] = {160, 120, 80, 40};
    PwmOut wave(p11);

    wave.period_ms(periods[result]);
    wave = 0.5;

}

