#include "mbed.h"
#include "FastPWM.h"

Serial COM(USBTX, USBRX);

DigitalOut batterie         (D4);
DigitalOut air_brake        (D5);
DigitalOut parking_brake    (D6);
DigitalOut oil              (D7);

DigitalOut dash_on          (A0);
DigitalOut light_on         (A1);
DigitalOut spliter_light    (A2);
DigitalOut blinkers         (A3);
DigitalOut retarder_light   (A4);
DigitalOut high_beam_light  (A5);

FastPWM fuel_pwm            (PA_11);
FastPWM water_pwm           (D2);
FastPWM air_pwm             (D3);
FastPWM oil_pwm             (D8);
FastPWM speed_pulse         (D9);
FastPWM batterie_neg        (D10);
FastPWM batterie_pos        (D11);
FastPWM buzzer              (D13);
FastPWM rpm_pulse           (D14);

int serial_byte;
int prev_rpm;
float prev_speed;

float map(float x, float in_min, float in_max, float out_min, float out_max)
{
  return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

int    Serial_data[41];
bool   OCTET0[8]; 
bool   OCTET1[8]; 
bool   OCTET2[8];

enum   OCTET0{brake_air_press_warning, park_brake, high_beam, low_beam, right_blinker, left_blinker, light_parking};
enum   OCTET1{water_temperature_warning, oil_pressure_warning,  batterie_voltage_warning, fuel_warning, brake_air_emergency, motor_brake, light_reverse};
enum   OCTET2{engine_enabled, electric_enabled, left_blinker_state, right_blinker_state, NOK1, light_brake,  trailer_connected};
enum   Serial_data {speed_ms, LSB_rpm, MSB_rpm, air_pressure, fuel_int, oil_pressure_int, oil_temperature_int, water_temperature_int, batterie_voltage_int, average_cosumption_int, adblue, engine_gear, effective_throttle, LSB_Fuel_liters, MSB_Fuel_liters, odometer_split_1,
                   odometer_split_2, odometer_split_3, odometer_split_4, game_time_split_1, game_time_split_2, game_time_split_3, game_time_split_4, next_rest_stop_split_1, next_rest_stop_split_2, next_rest_stop_split_3, next_rest_stop_split_4, cruise_control,
                   retarder_steps, decimal_oil_pressure, decimal_battery_voltage, decimal_average_consumption,  speed_ms_decimal, decimal_fuel_liters, decimal_odometer, decimal_water_temperature, decimal_fuel, decimal_engine_rpm, OCTET_0, OCTET_1, OCTET_2};

float fuel_level_needle[100]        ={0.97,0.9652,0.9604,0.9556,0.9508,0.946,0.9412,0.9364,0.9316,0.9268,0.922,0.9172,0.9124,0.9076,0.9028,0.898,0.8932,0.8884,0.8836,0.8788,0.874,0.8692,0.8644,0.8596,0.85,0.8448,0.8396,0.8344,0.8292,0.824,0.8188,0.8136,0.8084,0.8032,0.798,0.7928,0.7876,0.7824,0.7772,0.772,0.7668,0.7616,0.7564,0.7512,0.746,0.7408,0.7356,0.7304,0.7252,0.72,0.7168,0.7136,0.7104,0.7072,0.704,0.7008,0.6976,0.6944,0.6912,0.688,0.6848,0.6816,0.6784,0.6752,0.672,0.6688,0.6656,0.6624,0.6592,0.656,0.6528,0.6496,0.6464,0.6432,0.64,0.6384,0.6368,0.6352,0.6336,0.632,0.6304,0.6288,0.6272,0.6256,0.624,0.6224,0.6208,0.6192,0.6176,0.616,0.6144,0.6128,0.6112,0.6096,0.608,0.6064,0.6048,0.6032,0.6016,0.6};
float air_pressure_needle[150]      ={0.95,0.9424,0.9348,0.9272,0.9196,0.912,0.9044,0.8968,0.8892,0.8816,0.874,0.8664,0.8588,0.8512,0.8436,0.836,0.8284,0.8208,0.8132,0.8056,0.798,0.7904,0.7828,0.7752,0.76,0.7541,0.7482,0.7423,0.7364,0.7305,0.7246,0.7187,0.7128,0.7069,0.701,0.6951,0.6892,0.6833,0.6774,0.6715,0.6656,0.6597,0.6538,0.6479,0.642,0.6361,0.6302,0.63,0.6267,0.6234,0.6201,0.6168,0.6135,0.6102,0.6069,0.6036,0.6003,0.597,0.5937,0.5904,0.5871,0.5838,0.5805,0.5772,0.5739,0.5706,0.5673,0.564,0.5607,0.5574,0.5541,0.55,0.5484,0.5468,0.5452,0.5436,0.542,0.5404,0.5388,0.5372,0.5356,0.534,0.5324,0.5308,0.5292,0.5276,0.526,0.5244,0.5228,0.5212,0.5196,0.518,0.5164,0.5148,0.5132,0.51,0.50917,0.50834,0.50751,0.50668,0.50585,0.50502,0.50419,0.50336,0.50253,0.5017,0.50087,0.50004,0.49921,0.49838,0.49755,0.49672,0.49589,0.49506,0.49423,0.4934,0.49257,0.49174,0.49091,0.49,0.48917,0.48834,0.48751,0.48668,0.48585,0.48502,0.48419,0.48336,0.48253,0.4817,0.48087,0.48004,0.47921,0.47838,0.47755,0.47672,0.47589,0.47506,0.47423,0.4734,0.47257,0.47174,0.47091,0.47,0.47,0.47,0.47,0.47,0.47,0.47};
float water_temperature_needle[120] ={0.7,0.705,0.71,0.715,0.72,0.725,0.73,0.735,0.74,0.745,0.75,0.755,0.76,0.765,0.77,0.775,0.78,0.785,0.79,0.795,0.8,0.805,0.81,0.815,0.82,0.825,0.83,0.835,0.84,0.845,0.85,0.855,0.86,0.865,0.87,0.875,0.88,0.885,0.89,0.9,0.901,0.902,0.903,0.904,0.905,0.906,0.907,0.908,0.909,0.91,0.911,0.912,0.913,0.914,0.915,0.916,0.917,0.918,0.919,0.92,0.921,0.922,0.923,0.924,0.925,0.926,0.927,0.928,0.929,0.93,0.931,0.932,0.933,0.934,0.935,0.936,0.937,0.938,0.939,0.94,0.94135,0.9427,0.94405,0.9454,0.94675,0.9481,0.94945,0.9508,0.95215,0.9535,0.95485,0.9562,0.95755,0.9589,0.96025,0.9616,0.96295,0.9643,0.96565,0.967,0.968,0.969,0.97,0.971,0.972,0.973,0.974,0.975,0.976,0.977,0.978,0.979,0.98,0.981,0.982,0.983,0.984,0.985,0.986,0.987};
float oil_pressure_needle[86]       ={0.97,0.9637,0.9574,0.9511,0.9448,0.9385,0.9322,0.9259,0.9196,0.9133,0.9,0.8928,0.8856,0.8784,0.8712,0.864,0.8568,0.8496,0.8424,0.8352,0.828,0.82,0.8145,0.809,0.8035,0.798,0.7925,0.787,0.7815,0.776,0.7705,0.765,0.76,0.7563,0.7526,0.7489,0.7452,0.7415,0.7378,0.7341,0.7304,0.7267,0.723,0.72,0.7168,0.7136,0.7104,0.7072,0.704,0.7008,0.6976,0.6944,0.6912,0.688,0.685,0.6828,0.6806,0.6784,0.6762,0.674,0.6718,0.6696,0.6674,0.6652,0.663,0.6608,0.6586,0.6564,0.6542,0.652,0.6498,0.6476,0.6454,0.6432,0.641,0.6388,0.6366,0.6344,0.6322,0.63,0.63,0.63,0.63,0.63,0.63,0.63};

int main()
{
    COM.baud(115200);
    COM.format(8, SerialBase::None, 1);
    
    //initialisation...
    dash_on                      = 1;
    light_on                     = 0;
    batterie                     = 0;
    oil                          = 0;
    parking_brake                = 0;
    air_brake                    = 0;
    blinkers                     = 0;
    retarder_light               = 0;
    high_beam_light              = 0;
    spliter_light                = 0;
 
    //speed set at 0KM/H
    speed_pulse.period_us   (200000);
    speed_pulse.write        (0.50f);
  
    //set water, baterie(-), oil PWM frequency at 887Hz (timer1 linked)
    water_pwm.period_us        (200); // 887Hz 
    water_pwm.write          (0.30f);
    oil_pwm.write            (0.97f);
  
    //buzzer configuration...frequency set for air,fuel, buzzer PWM (timer2 linked)
    buzzer.period_us          (2257);  // 887Hz
    buzzer.write              (0.0f);
  
    fuel_pwm.write           (0.97f);
    air_pwm.write            (0.95f);
    
    while(1) {


        if(COM.readable()) {
            
            serial_byte = COM.getc();

            if(serial_byte != 0xFF) {
                return;
            }

            serial_byte = COM.getc();

            if(serial_byte != 0x02) {
                return;
            }
            
            for (int i = 0; i<41; i++)
            {
                Serial_data[i] = COM.getc();
            }

        for(int i = 0; i < 8; i ++)
        {
            OCTET0[i] = ((int (Serial_data[OCTET_0]) >> i) & 0x01);
            OCTET1[i] = ((int (Serial_data[OCTET_1]) >> i) & 0x01);
            OCTET2[i] = ((int (Serial_data[OCTET_2]) >> i) & 0x01);
        }
            

      float rpm                                   = ((Serial_data[MSB_rpm] << 8) + Serial_data[LSB_rpm]) + (decimal_engine_rpm);
      float speed_kph                             = (Serial_data[speed_ms] + (Serial_data[speed_ms_decimal] / 100.0))*3.60f;
     
      if (Serial_data[retarder_steps] > 1) 
            Serial_data[retarder_steps] = 1;

           

//-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

            light_on        = OCTET0[light_parking];
            dash_on         = OCTET2[electric_enabled];
            batterie        = OCTET1[batterie_voltage_warning];
            oil             = OCTET1[oil_pressure_warning];
            parking_brake   = OCTET0[park_brake];
            air_brake       = OCTET0[brake_air_press_warning];
            blinkers        = (OCTET0[left_blinker] || OCTET0[right_blinker]);
            retarder_light  = Serial_data[retarder_steps];
            high_beam_light = OCTET0[high_beam];
            spliter_light   = (Serial_data[engine_gear] & 0x01);
            
            if(OCTET0[brake_air_press_warning] == 1){
                buzzer.write(0.50f);
                }else{
                    buzzer.write(0);
                    }
           
            oil_pwm.write       (oil_pressure_needle[Serial_data[oil_pressure_int]]);
            air_pwm.write       (air_pressure_needle[Serial_data[air_pressure]]); 
            fuel_pwm.write      (fuel_level_needle[Serial_data[fuel_int]]); 
            water_pwm.write     (water_temperature_needle[Serial_data[water_temperature_int]]);
            batterie_neg.write  (0);
            batterie_pos.write  (map(Serial_data[batterie_voltage_int], 0.0, 25.0, 0.00, 1.00) && OCTET2[electric_enabled]);
            
            float period;
            
            if ((rpm > 200) && (rpm != prev_rpm)){
                prev_rpm = rpm;
                rpm = map(rpm, 200.0, 3000.0, 20.0, 1030.0);  
                period = (1.000000f/rpm);
                rpm_pulse.period_us(double(period)*1000000);
                rpm_pulse.write(0.50f); 
                 
            } 
            
            if(engine_gear == 13 || engine_gear == 14){
                speed_kph = 0.0f;
                }
            
            if(speed_kph != prev_speed){
                prev_speed = speed_kph;
                 speed_kph = map(speed_kph, 0, 120, 5, 160);  
                period = (1.000000f/speed_kph);
                speed_pulse.period_us(double(period)*1000000);
                speed_pulse.write(0.50f);
                }


        }
    }
}
