/* mbed CD Library, for a i2c LCD based on PCF2119
 * Copyright (c) 2016, aracosta, http://mbed.org
 * Copyright (c) 2007-2010, sford, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "pcf2119.h"
#include "mbed.h"

extern Serial pc;

TextLCD::TextLCD(void) : _i2c(NC, NC), _reset(NC)
{
    return;
}

TextLCD::TextLCD(PinName pSDA, PinName pSCL, PinName pReset, LCDType type) : _i2c(pSDA, pSCL), _reset(pReset), _type(type) {  
    if (_type != LCDuser)
        setLCDparam(_type); // otherwise rows, colums, comdelay, adresses must be set before


    _reset = 1;
    wait(0.05f);
    _reset = 0;
    wait(0.05f);
    _reset = 1;
    wait(0.05f);
    _reset = 0;
    
    wait(0.2f);       // Wait 200ms to ensure powered up
  
    writeCommand(0x24); // Function set 4 Bit, 2Line, 5*7
    cls();              // clear display, reset _column and _row
    writeCommand(0x0c); // Display on
    writeCommand(0x06); // cursor right, Display is not shifted
}

void TextLCD::character(int column, int row, int c) {
    int a = 0x80 | (LCDparam.addresses[row & 3] + column);
    writeCommand(a);    // set cursor address
    writeData(c + 0x80);       // write char
}

void TextLCD::cls() {
    writeCommand(0x08); // Display on
    int a = 0x80 | (LCDparam.addresses[0 & 3] + 0);
    writeCommand(a);    // set cursor address
    writeCommand(0x80); // set cursor address
    for( int i=0;i<LCDparam.columns;i++) {
        writeData(' ' + 0x80);       // write char
    }
    a = 0x80 | (LCDparam.addresses[1 & 3] + 0);
    writeCommand(a);    // set cursor address
    for( int i=0;i<LCDparam.columns;i++) {
        writeData(' ' + 0x80);       // write char
    }
    locate(0, 0);       // set internal position
    writeCommand(0x0c); // Display on
    wait_us(45 * LCDparam.delay);   // CLS need much time
}

void TextLCD::locate(int column, int row) {
    _column = column;   // set position for next char
    _row = row;         // note: cursor is not set yet
}

int TextLCD::_putc(int value) {
    if (value == '\n') {
        _column = 0;
        _row++;
        if (_row >= LCDparam.rows) {
            _row = 0;
        }
    } else {
        character(_column, _row, value);
        _column++;
        if (_column >= LCDparam.columns) {
            _column = 0;
            _row++;
            if (_row >= LCDparam.rows) {
                _row = 0;
            }
        }
    }
    return value;
}

// Dummy function - read not supported
int TextLCD::_getc() {
    return -1;
}

void TextLCD::writeCommand(int command) {
    _i2cbuffer[0] = 0x00;
    _i2cbuffer[1] = command;
    _i2c.write(LCDparam.i2c_address, _i2cbuffer, 2, false);
    wait_us(LCDparam.delay);
}


void TextLCD::writeData(int data) {
    _i2cbuffer[0] = 0x40;
    _i2cbuffer[1] = data;
    _i2c.write(LCDparam.i2c_address, _i2cbuffer, 2, false);
    wait_us(LCDparam.delay);
}


// set user defined char 
void  TextLCD::writeCGRAM(int address, int pattern[8]){
    int i;
    address = address & 7;  //max 8 char
    for(i=0;i<8;i++){
        writeCommand(0x40 | (address * 8) + i);
        writeData(pattern[i]);
    }
}   

void TextLCD::setLCDparam(LCDType _type){
    switch (_type) {
        case LCD16x2:
        case LCD16x2B:
            LCDparam.columns = 16;
            break;
        case LCD20x2:
        case LCD20x4:
            LCDparam.columns = 20;
            break;
        case LCD24x2:
            LCDparam.columns = 24;
            break;
    }
    if (_type == LCD20x4) 
        LCDparam.rows = 4;
    else 
        LCDparam.rows = 2;
        
    LCDparam.addresses[0] = 0;
    
    if (_type == LCD16x2B)
        LCDparam.addresses[1] = 40;
    else
        LCDparam.addresses[1] = 0x40;
        
    if (_type == LCD20x4) {
        LCDparam.addresses[2] = 0x14;
        LCDparam.addresses[3] = 0x54;}
    else {
        LCDparam.addresses[2] = 0;
        LCDparam.addresses[3] = 0;}
        
    LCDparam.delay = 50;            // 50 us delays as default       
    
    LCDparam.i2c_address = 0x76;       
}
        
