#ifndef MBED_SPI_TFT_H
#define MBED_SPI_TFT_H


#include "mbed.h"
#include "GraphicsDisplay.h"

extern "C" void mbed_reset();

#define RGB(r,g,b)  (((r&0xF8)<<8)|((g&0xFC)<<3)|((b&0xF8)>>3)) //5 red | 6 green | 5 blue

#define SPI_START   (0x70)              /* Start byte for SPI transfer        */
#define SPI_RD      (0x01)              /* WR bit 1 within start              */
#define SPI_WR      (0x00)              /* WR bit 0 within start              */
#define SPI_DATA    (0x02)              /* RS bit 1 within start byte         */
#define SPI_INDEX   (0x00)              /* RS bit 0 within start byte         */


/* some RGB color definitions                                                
# Define RGB565CONVERT NEGRO ( 0 ,     0 ,       0 )
# Define RGB565CONVERT BRIGHTBLUE ( 0 ,     0 ,       255 )
# Define RGB565CONVERT BRIGHTGREEN ( 0 ,     255 ,     0 )
# Define RGB565CONVERT BRIGHTCYAN ( 0 ,     255 ,     255 )
# Define RGB565CONVERT BRIGHTRED ( 255 ,   0 ,       0 )
# Define RGB565CONVERT BRIGHTMAGENTA ( 255 ,   0 ,       255 )
# Define RGB565CONVERT BRIGHTYELLOW ( 255 ,   255 ,     0 )
# Define RGB565CONVERT AZUL ( 0 ,     0 ,       128 )
# Define RGB565CONVERT VERDE ( 0 ,     128 ,     0 )
# Define RGB565CONVERT CIAN ( 0 ,     128 ,     128 )
# Define RGB565CONVERT ROJO ( 128 ,   0 ,       0 )
# Define MAGENTA RGB565CONVERT ( 128 ,   0 ,       128 )
# Define RGB565CONVERT BROWN ( 255 ,   128 ,     0 )
# Define RGB565CONVERT gris_claro ( 128 ,   128 ,     128 )
# Define RGB565CONVERT gris_oscuro ( 64 ,    64 ,      64 )
# Define RGB565CONVERT lightblue ( 128 ,   128 ,     255 )
# Define RGB565CONVERT verde claro ( 128 ,   255 ,     128 )
# Define RGB565CONVERT LIGHTCYAN ( 128 ,   255 ,     255 )
# Define RGB565CONVERT LIGHTRED ( 255 ,   128 ,     128 )
# Define RGB565CONVERT LIGHTMAGENTA ( 255 ,   128 ,     255 )
# Define RGB565CONVERT AMARILLO ( 255 ,   255 ,     128 )
# Define RGB565CONVERT BLANCO ( 255 ,   255 ,     255 )
                            
# Define GRAY0 RGB565CONVERT ( 224 ,   224 ,     224 )
# Define GRAY1 RGB565CONVERT ( 192 ,   192 ,     192 )   
# Define GRAY2 RGB565CONVERT ( 160 ,   160 ,     160 )   
# Define GRAY3 RGB565CONVERT ( 128 ,   128 ,     128 )
# Define GRAY4 RGB565CONVERT ( 96 ,    96 ,      96 )
# Define GRAY5 RGB565CONVERT ( 64 ,    64 ,      64 )
# Define GRAY6 RGB565CONVERT ( 32 ,    32 ,      32 )

#define GRAY20      RGB565CONVERT(51, 51, 51)
#define GRAY40      RGB565CONVERT(102, 102, 102)
#define GRAY80      RGB565CONVERT(204, 204, 204)
#define GRAY90      RGB565CONVERT(229, 229, 229)
#define GRAY95      RGB565CONVERT(242, 242, 242)
#define RED4        RGB565CONVERT(139, 0, 0)
#define FIREBRICK1  RGB565CONVERT(255, 48, 48)
#define DARKGREEN   RGB565CONVERT(0, 100, 0)
#define PALEGREEN   RGB565CONVERT(152, 251, 152)
#define LIGHTYELLOW RGB565CONVERT(238, 221, 130)
#define GOLD        RGB565CONVERT(255, 215, 0)
#define DARKORANGE  RGB565CONVERT(255, 140, 0)
 */
#define Black           0x0000      /*   0,   0,   0 */
#define Navy            0x000F      /*   0,   0, 128 */
#define DarkGreen       0x03E0      /*   0, 128,   0 */
#define DarkCyan        0x03EF      /*   0, 128, 128 */
#define Maroon          0x7800      /* 128,   0,   0 */
#define Purple          0x780F      /* 128,   0, 128 */
#define Olive           0x7BE0      /* 128, 128,   0 */
#define LightGrey       0xC618      /* 192, 192, 192 */
#define DarkGrey        0x7BEF      /* 128, 128, 128 */
#define Blue            0x001F      /*   0,   0, 255 *///////////////////////////   32
#define Green           0x07E0      /*   0, 255,   0 *///////////////////////////   2016
#define Cyan            0x07FF      /*   0, 255, 255 */
#define Red             0xF800      /* 255,   0,   0 *///////////////////////////   63488
#define Magenta         0xF81F      /* 255,   0, 255 */
#define Yellow          0xFFE0      /* 255, 255,   0 */
#define White           0xFFFF      /* 255, 255, 255 */
#define Orange          0xFD20      /* 255, 165,   0 */
#define GreenYellow     0xAFE5      /* 173, 255,  47 */


/** Display control class, based on GraphicsDisplay and TextDisplay
 *
 * Example:
 * @code
 * #include "stdio.h"
 * #include "mbed.h"
 * #include "SPI_TFT.h"
 * #include "string"
 * #include "Arial12x12.h"
 * #include "Arial24x23.h"
 * 
 *
 *
 * // the TFT is connected to SPI pin 5-7 
 * SPI_TFT TFT(p5, p6, p7, p8, p15,"TFT"); // mosi, miso, sclk, cs, reset
 * 
 * int main() {
 *     TFT.claim(stdout);      // send stdout to the TFT display 
 *     //TFT.claim(stderr);      // send stderr to the TFT display
 *
 *     TFT.background(Black);    // set background to black
 *     TFT.foreground(White);    // set chars to white
 *     TFT.cls();                // clear the screen
 *     TFT.set_font((unsigned char*) Arial12x12);  // select the font
 *     
 *     TFT.set_orientation(0);
 *     TFT.locate(0,0);
 *     printf("  Hello Mbed 0");
 *     TFT.set_font((unsigned char*) Arial24x23);  // select font 2
 *     TFT.locate(48,115);
 *     TFT.printf("Bigger Font");
 *  }
 * @endcode
 */
 class SPI_TFT : public GraphicsDisplay {
 public:

  /** Create a SPI_TFT object connected to SPI and two pins
   *
   * @param mosi,miso,sclk SPI
   * @param cs pin connected to CS of display
   * @param reset pin connected to RESET of display
   *
   */ 
  SPI_TFT(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName reset, PinName bk,const char* name ="TFT");
    
  /** Get the width of the screen in pixel
   *
   * @param 
   * @returns width of screen in pixel
   *
   */    
  virtual int width();

  /** Get the height of the screen in pixel
   *
   * @returns height of screen in pixel 
   *
   */     
  virtual int height();
    
  /** Draw a pixel at x,y with color 
   *  
   * @param x horizontal position
   * @param y vertical position
   * @param color 16 bit pixel color
   */    
  virtual void pixel(int x, int y, int colour);
    
  /** draw a circle
   *
   * @param x0,y0 center
   * @param r radius
   * @param color 16 bit color                                                                 *
   *
   */    
  void circle(int x, int y, int r, int colour); 
  
  /** draw a filled circle
   *
   * @param x0,y0 center
   * @param r radius
   * @param color 16 bit color                                                                 *
   *
   * use circle with different radius,
   * can miss some pixel
   */    
  void fillcircle(int x, int y, int r, int colour); 
  
 
    
  /** draw a 1 pixel line
   *
   * @param x0,y0 start point
   * @param x1,y1 stop point
   * @param color 16 bit color
   *
   */    
  void line(int x0, int y0, int x1, int y1, int colour);
    
  /** draw a rect
   *
   * @param x0,y0 top left corner
   * @param x1,y1 down right corner
   * @param color 16 bit color
   *                                                   *
   */    
  void rect(int x0, int y0, int x1, int y1, int colour);
    
  /** draw a filled rect
   *
   * @param x0,y0 top left corner
   * @param x1,y1 down right corner
   * @param color 16 bit color
   *
   */    
  void fillrect(int x0, int y0, int x1, int y1, int colour);
    
  /** setup cursor position
   *
   * @param x x-position (top left)
   * @param y y-position 
   */ 
     
  virtual void locate(int x, int y);
    
  /** Fill the screen with _backgroun color
   *
   */ 
     
  virtual void cls (void);   
    
  /** calculate the max number of char in a line
   *
   * @returns max columns
   * depends on actual font size
   *
   */ 
      
  virtual int columns(void);
    
  /** calculate the max number of columns
   *
   * @returns max column
   * depends on actual font size
   *
   */   

  virtual int rows(void);
    
  /** put a char on the screen
   *
   * @param value char to print
   * @returns printed char
   *
   */

  virtual int _putc(int value);
    
  /** draw a character on given position out of the active font to the TFT
   *
   * @param x x-position of char (top left) 
   * @param y y-position
   * @param c char to print
   *
   */   
    
  virtual void character(int x, int y, int c);
    
  /** paint a bitmap on the TFT 
   *
   * @param x,y : upper left corner 
   * @param w width of bitmap
   * @param h high of bitmap
   * @param *bitmap pointer to the bitmap data
   *
   *   bitmap format: 16 bit R5 G6 B5
   * 
   *   use Gimp to create / load , save as BMP, option 16 bit R5 G6 B5            
   *   use winhex to load this file and mark data stating at offset 0x46 to end
   *   use edit -> copy block -> C Source to export C array
   *   paste this array into your program
   * 
   *   define the array as static const unsigned char to put it into flash memory
   *   cast the pointer to (unsigned char *) :
   *   tft.Bitmap(10,40,309,50,(unsigned char *)scala);
   */  
     
  void Bitmap(unsigned int x, unsigned int y, unsigned int w, unsigned int h,unsigned char *bitmap);
    
    
   /** paint a 16 bit BMP from local filesytem on the TFT (slow) 
   *
   * @param x,y : upper left corner 
   * @param *Name_BMP name of the BMP file
   * @returns 1 if bmp file was found and painted
   * @returns -1 if bmp file was found not found
   * @returns -2 if bmp file is not 16bit
   * @returns -3 if bmp file is to big for screen 
   * @returns -4 if buffer malloc go wrong
   *
   *   bitmap format: 16 bit R5 G6 B5
   * 
   *   use Gimp to create / load , save as BMP, option 16 bit R5 G6 B5
   *   copy to internal file system            
   * 
   */      
    
  int BMP_16(unsigned int x, unsigned int y, const char *Name_BMP);  
   
   
      /** paint a 16 bit BMP from memory uSD Card on the TFT (slow) 
   *
   * @param x,y : upper left corner 
   * @param *Name_BMP name of the BMP file
   * @returns 1 if bmp file was found and painted
   * @returns -1 if bmp file was found not found
   * @returns -2 if bmp file is not 16bit
   * @returns -3 if bmp file is to big for screen 
   * @returns -4 if buffer malloc go wrong
   *
   *   bitmap format: 16 bit R5 G6 B5
   * 
   *   use Gimp to create / load , save as BMP, option 16 bit R5 G6 B5
   *   copy to internal file system            
   * 
   */      
    
  int BMP_SD(unsigned int x_sd, unsigned int y_sd, const char *Name_BMP_SD); 
    
    
  /** select the font to use
   *
   * @param f pointer to font array 
   *                                                                              
   *   font array can created with GLCD Font Creator from http://www.mikroe.com
   *   you have to add 4 parameter at the beginning of the font array to use: 
   *   - the number of byte / char
   *   - the vertial size in pixel
   *   - the horizontal size in pixel
   *   - the number of byte per vertical line
   *   you also have to change the array to char[]
   *
   */  

  void set_font(unsigned char* f);
   
  /** Set the orientation of the screen
   *  x,y: 0,0 is always top left 
   *
   * @param o direction to use the screen (0-3) 90&#65533; Steps  
   *
   */  

  void set_orientation(unsigned int o);
    
  SPI _spi;
  DigitalOut _cs;
  DigitalOut _reset;
  PwmOut _bk;
  unsigned char* font;
  
  /**Resetea mediante hardware el LFCD TFT a travez del pin RESET
  */

  void tft_rst_hw();
  
  /**Enciende el Backligth del LCD TFT a travez de un pin PWM
  */

  void backlight(float intensity);  
  
  /**Modula el Backligth del LCD TFT a travez de un pin PWM
  * minimo 0.01; maximo 1.0 ; paso para FOR 0.01, retardo 0.01
  */

  void contraste_pos(float min, float max, float steep, float delay);
  
  /**Modula el Backligth del LCD TFT a travez de un pin PWM
  * minimo 0.01; maximo 1.0 ; paso para FOR 0.01, retardo 0.01
  */

  void contraste_neg(float max, float min, float steep, float delay);
  
  /* Genera una barra de carga para animar la congiguracion de un programa
  * mientras este lee un archivo *.cfg de la raiz del sistema y configura las 
  * operaciones generales del programa segun indique el estado de su
  * configuracion guardado la anterior vez en su memoria
  */
  void charge(int x0, int y0, int x1, int y1, int color_max, int color_min, float speed);
  
  /* Animacion controlada por PWM para parpadeos de la pantalla, estipular el numero de parpadeos
  *  el maximo de intensidad, el minimo, la cantidad de incremento en steep, el retardo por cada uno
  *  el tiempo de parada entre cada uno y el brillo final que mantendra la LCD TFT al terminar la animacion
  */
  
  void parpadeo(int parpadeos, float max, float min, float steep, float delay,float stop, float brigthness);
  
  /*
  *
  */
  void Init_Kernel_Up(float brillo, int color, int orientation, unsigned char* letra,int x,int y,int delay);
  
  /*
  *
  */
  void Init_Kernel_Down(float brillo, int color, int orientation, unsigned char* letra,int x,int y,int delay);
    
  /*
  *
  */
  
  void init_tasking(float brillo, int color, int orientation, unsigned char* letra,int x,int y,int delay);
  
   /*
  *
  */
  void strcmp(float brillo, int color, int orientation, unsigned char* letra,int x,int y,int delay);
 
  /*
  *
  */
  void Loading(float bkt,int fondo, int color, unsigned char* letra,const char *titulo1,int x,int y,int fondo2, int color2,unsigned char* sam, const char *titulo2, int z,int w, int delay,int a,int b, int c, int d, int max, int min, float alfa_cromatic);


 /*trae el codigo del LCD TFT para identificarlo mediante el registro 0x00h
 *
 */
  int ID();
 
  void binary_init(int a, int b, int c, int d, int delay); 
  
/*permite definir el espacio maximo de desplazamiento para la pantalla
 * top es TFA Top Fixed Area, button es BFA, button fixed area, y height es Height de Vertical Scroll Area
 *  top = 0, button= 0, height = 320, definen la pantalla maxima, top es el inicio y button es el resto, la suma de
 *  t+b+h = 320, si esto no se cumple el sroll no estaria bien definido.
 *  p es el puntero incremental o decremental que posicionara la imagen 
 *  si pointer = 30, aparecera desplazado 30 pixeles   
 *  speed es la velocidad a la que la imagen se desplaza
 */
 void sharepoint(unsigned int top, unsigned int button, unsigned int height, unsigned int pointer, unsigned int speed, unsigned repetitions);
 void icon_ball(int x, int y, int state);

/*****************************************************************************************************/
  

  
//protected:
    /** Init the HX8347D controller 
   *
   */    
   
   
  void tft_init();

  /** Set draw window region to whole screen
   *
   */ 
    
  void WindowMax (void);


  /** draw a horizontal line
   *
   * @param x0 horizontal start
   * @param x1 horizontal stop
   * @param y vertical position
   * @param color 16 bit color                                               
   *
   */

  void hline(int x0, int x1, int y, int colour);
    
  /** draw a vertical line
   *
   * @param x horizontal position
   * @param y0 vertical start 
   * @param y1 vertical stop
   * @param color 16 bit color
   */

  void vline(int y0, int y1, int x, int colour);
    
  /** Set draw window region
   *
   * @param x horizontal position
   * @param y vertical position
   * @param w window width in pixel
   * @param h window height in pixels
   */ 
      
  virtual void windows (unsigned int x, unsigned int y, unsigned int w, unsigned int h);
    
 
    

    
   /** Write data to the LCD controller
   *
   * @param dat data written to LCD controller
   * 
   */ 
     
  void wr_dat(int value);
    
  /** Write a command the LCD controller 
   *
   * @param cmd: command to be written   
   *
   */  
    
  void wr_cmd(int value);
    
   /** Start data sequence to the LCD controller
   * 
   */  
    
  void wr_dat_start();
    
  /** Stop of data writing to the LCD controller
   *   
   */  

  void wr_dat_stop();
    
  /** write data to the LCD controller
   *
   * @param data to be written 
   *                                           *
   */  
     
  void wr_dat_only(unsigned short dat);
    
  /** Read data from the LCD controller
   *
   * @returns data from LCD controller
   *  
   */ 
      
  unsigned short rd_dat(void);
    
  /** Write a value to the to a LCD register
   *
   * @param reg register to be written
   * @param val data to be written
   */
      
  void wr_reg (unsigned char reg, unsigned short val);
    
  /** Read a LCD register
   *
   * @param reg register to be read
   * @returns value of the register 
   */ 
      
  unsigned short rd_reg (unsigned char reg);
    
  unsigned int orientation;
  unsigned int char_x;
  unsigned int char_y;
  
  protected:
 
    
};

#endif
