#include "SPI_TFT.h"
#include "mbed.h"


#define BPP         16                  // Bits per pixel                

SPI_TFT::SPI_TFT(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName reset, PinName bk, const char *name)
        :_spi(mosi, miso, sclk), _cs(cs), _reset(reset), _bk(bk), GraphicsDisplay(name) {                                 //Constructor
    //tft_rst_hw();
    tft_init();
    backlight(0);
    orientation = 3;
    char_x = 0;
    backlight(0);  //0.0001
    foreground(Black);
    background(Black);

}

int SPI_TFT::width() {
    if (orientation == 0 || orientation == 2) return 240;
    else return 320;
}


int SPI_TFT::height() {
    if (orientation == 0 || orientation == 2) return 320;
    else return 240;
}



void SPI_TFT::set_orientation(unsigned int x) {
    orientation = x;
    switch (orientation) {
        case 0:
            wr_reg(0x16, 0x0008);//240x320
            break;
        case 1:
            wr_reg(0x16, 0x0068);//320x240
            break;
        case 2:
            wr_reg(0x16, 0x00C8);//240x320
            break;
        case 3:
            wr_reg(0x16, 0x00A8);//320x240
            break;
    }
}


//configura el registro index enviando para ello: (01110000+00000000+00000000)=01110000
void SPI_TFT::wr_cmd(int cmd) {
    _cs = 0;
    _spi.write(SPI_START | SPI_WR | SPI_INDEX);   /* Write : RS = 0, RW = 0   */
    _spi.write(cmd);
    _cs = 1;
}


//escribe instrucciones o datos en GRAM enviando para ello: (01110000+00000000+00000010)=01110010
void SPI_TFT::wr_dat(int dat) {
    _cs = 0;
    _spi.write(SPI_START | SPI_WR | SPI_DATA);    // Write : RS = 1, RW = 0
    _spi.format(16,3);                            // switch to 16 bit Mode 3
    _spi.write(dat);                              // Write D0..D15
    _spi.format(16,3);                             // 8 bit Mode 3
    _cs = 1;
}


//escribe instrucciones o datos en GRAM enviando para ello: (01110000+00000000+00000010)=01110010
void SPI_TFT::wr_dat_start(void) {
    _cs = 0;
    _spi.write(SPI_START | SPI_WR | SPI_DATA);    /* Write : RS = 1, RW = 0       */  //70-00-02
}


//detiene la secuencia de envio y la comunicacion con el TFT
void SPI_TFT::wr_dat_stop (void) {
    _cs = 1;
}


//envia un dato simple
void SPI_TFT::wr_dat_only (unsigned short dat) {

    _spi.format(16,3);                        // switch to 16 bit Mode 3
    _spi.write(dat);                          // Write D0..D15
    _spi.format(16,3);                         // 8 bit Mode 3
}


//lee comandos (16bits en total) pero no soporta lectura de GRAM enviando para ello: (01110000+00000001+00000010)=01110011
unsigned short SPI_TFT::rd_dat (void) {
    unsigned short val = 0;

    _cs = 0;
    _spi.write(SPI_START | SPI_RD | SPI_DATA);    /* Read: RS = 1, RW = 1         */
    _spi.write(0);                                /* Dummy read 1                 */ //lectura ficticia
    val   = _spi.write(0);                        /* Read D8..D15                 */ //lee los altos y luego los bajos
    val <<= 8;
    val  |= _spi.write(0);                        /* Read D0..D7                  */
    _cs = 1;
    return (val);//devuelve la concatenacion completa de 16bits
}



void SPI_TFT::wr_reg (unsigned char reg, unsigned short val) {

    wr_cmd(reg);
    wr_dat(val);
}



unsigned short SPI_TFT::rd_reg (unsigned char reg) {

    wr_cmd(reg);
    return(rd_dat());
}

int SPI_TFT::ID() {
    static unsigned short driverCode;
    driverCode = rd_reg(0x01);        // read controller ID
    //printf("Disp_ID = %x debe ser 47h",driverCode);
    //pc.printf("Disp_ID = %x debe ser 47h",tt.ID());
    return driverCode;
}

void SPI_TFT::tft_init() {
    //static unsigned short driverCode;
    _spi.format(16,3);                 // 8 bit spi mode 3
    _spi.frequency(100000000);         // 48Mhz SPI clock
    _reset = 0;                       // reset
    _cs = 1;
    wait_us(50);
    _reset = 1;                       // end reset
    wait_ms(6);//5
    /*
    driverCode = rd_reg(0x00);        // read controller ID
    printf("Disp_ID = %x debe ser 47h",driverCode);
    wait_ms(50);
    */

    //              SECUENCIA DE INICIO Y DEFINICION DE CONFIGURACIONES GENERALES                             //

    /* Start Initial Sequence ----------------------------------------------------*/
    wr_reg(0xEA, 0x0000);                 /* Reset Power Control 1                */
    wr_reg(0xEB, 0x0020);                 /* Power Control 2                      */
    wr_reg(0xEC, 0x000C);                 /* Power Control 3                      */
    wr_reg(0xED, 0x00C4);                 /* Power Control 4                      */
    wr_reg(0xE8, 0x0040);                 /* Source OPON_N                        */
    wr_reg(0xE9, 0x0038);                 /* Source OPON_I                        */
    wr_reg(0xF1, 0x0001);                 /*                                      */
    wr_reg(0xF2, 0x0010);                 /*                                      */
    wr_reg(0x27, 0x00A3);                 /* Display Control 2                    */


    /* Gamma settings  -----------------------------------------------------------*/
    wr_reg(0x40,0x00);   //
    wr_reg(0x41,0x00);   //
    wr_reg(0x42,0x01);   //
    wr_reg(0x43,0x12);   //t12
    wr_reg(0x44,0x10);   //
    wr_reg(0x45,0x26);   //
    wr_reg(0x46,0x08);   //
    wr_reg(0x47,0x53);   //t53
    wr_reg(0x48,0x02);   //
    wr_reg(0x49,0x15);   //t15
    wr_reg(0x4A,0x19);   //t19
    wr_reg(0x4B,0x19);   //
    wr_reg(0x4C,0x16);   //t16
    wr_reg(0x50,0x19);   //
    wr_reg(0x51,0x2F);   //
    wr_reg(0x52,0x2D);   //t2D
    wr_reg(0x53,0x3E);   //
    wr_reg(0x54,0x3F);   //
    wr_reg(0x55,0x3F);   //
    wr_reg(0x56,0x2C);   //t2C
    wr_reg(0x57,0x77);   //
    wr_reg(0x58,0x09);   //t09
    wr_reg(0x59,0x06);   //
    wr_reg(0x5A,0x06);   //t06
    wr_reg(0x5B,0x0A);   //t0A
    wr_reg(0x5C,0x1D);   //
    wr_reg(0x5D,0xCC);   //

    /* Power On sequence ---------------------------------------------------------*/
    wr_reg(0x1B, 0x001B);                 /* Power Control 2                      *///VRH=4.65V
    wr_reg(0x1A, 0x0001);                 /* Power Control 1                      *///BT (VGH~15V,VGL~-10V,DDVDH~5V)
    wr_reg(0x24, 0x002F);                 /* Vcom Control 2                       *///VMH(VCOM High voltage ~3.2V)
    wr_reg(0x25, 0x0057);                 /* Vcom Control 3                       *///VML(VCOM Low voltage -1.2V)

    //****VCOM offset**///
    wr_reg(0x23, 0x0097);//8D);                 /* Vcom Control 1                       *///for Flicker adjust //can reload from OTP


    /* Power + Osc ---------------------------------------------------------------*/
    wr_reg(0x18, 0x003f);                 /* OSC Control 1                        */ //I/P_RADJ,N/P_RADJ, Normal mode 75Hz
    wr_reg(0x19, 0x0001);                 /* OSC Control 2                        */ //OSC_EN='1', start Osc
    wr_reg(0x01, 0x0009);//scroll activado               /* Display Mode Control                 */ //DP_STB='0', out deep sleep
    wr_reg(0x1F, 0x0088);                 /* Power Control 6                      */ // GAS=1, VOMG=00, PON=0, DK=1, XDK=0, DVDH_TRI=0, STB=0
    wait_ms(5);                           /* Delay 5 ms                           */
    wr_reg(0x1F, 0x0080);                 /* Power Control 6                      */ // GAS=1, VOMG=00, PON=0, DK=0, XDK=0, DVDH_TRI=0, STB=0
    wait_ms(5);                           /* Delay 5 ms                           */
    wr_reg(0x1F, 0x0090);                 /* Power Control 6                      */ // GAS=1, VOMG=00, PON=1, DK=0, XDK=0, DVDH_TRI=0, STB=0
    wait_ms(5);                           /* Delay 5 ms                           */
    wr_reg(0x1F, 0x00D0);                 /* Power Control 6                      */ // GAS=1, VOMG=10, PON=1, DK=0, XDK=0, DDVDH_TRI=0, STB=0
    wait_ms(5);                           /* Delay 5 ms                           */

    wr_reg(0x17, 0x0005);//5                 /* Colmod 16Bit/Pixel                   */ //default 0x0006 262k color // 0x0005 65k color

    wr_reg(0x36, 0x0000);                 /* Panel Characteristic                 */ //SS_P, GS_P,REV_P,BGR_P
    wr_reg(0x28, 0x0038);                 /* Display Control 3                    */ //GON=1, DTE=1, D=1000
    wait_ms(40);
    wr_reg(0x28, 0x003C);                 /* Display Control 3                    */ //GON=1, DTE=1, D=1100



    switch (orientation) {
        case 0:
            wr_reg(0x16, 0x0008);
            break;
        case 1:
            wr_reg(0x16, 0x0068);
            break;
        case 2:
            wr_reg(0x16, 0x00C8);
            break;
        case 3:
            wr_reg(0x16, 0x00A8);
            break;
    }

    WindowMax ();
}




void SPI_TFT::pixel(int x, int y, int color) {
    wr_reg(0x03, (x >> 0));
    wr_reg(0x02, (x >> 8));
    wr_reg(0x07, (y >> 0));
    wr_reg(0x06, (y >> 8));
    //wr_reg(0x05, (x+1 >> 0));
    //wr_reg(0x04, (x+1 >> 8));
    //wr_reg(0x09, (y+1 >> 0));
    //wr_reg(0x08, (y+1 >> 8));
    wr_cmd(0x22);
    wr_dat(color);
}




void SPI_TFT::windows (unsigned int x, unsigned int y, unsigned int w, unsigned int h) {
    wr_reg(0x03, (x >> 0));
    wr_reg(0x02, (x >> 8));
    wr_reg(0x05, (x+w-1 >> 0));
    wr_reg(0x04, (x+w-1 >> 8));
    wr_reg(0x07, ( y >> 0));
    wr_reg(0x06, ( y >> 8));
    wr_reg(0x09, ( y+h-1 >> 0));
    wr_reg(0x08, ( y+h-1 >> 8));
    //wr_cmd(0x22);
}


void SPI_TFT::WindowMax (void) {
    windows (0, 0, width(),  height());
}


void SPI_TFT::cls (void) {
    unsigned int i;
    WindowMax();
    wr_cmd(0x22);
    wr_dat_start();
    _spi.format(16,3);         // 16 bit Mode 3
    for (i = 0; i < ( width() * height()); i++)
        _spi.write(_background);
    _spi.format(16,3);         // 8 bit Mode 3
    wr_dat_stop();
}


void SPI_TFT::circle(int x0, int y0, int r, int color) {

    int draw_x0, draw_y0;
    int draw_x1, draw_y1;
    int draw_x2, draw_y2;
    int draw_x3, draw_y3;
    int draw_x4, draw_y4;
    int draw_x5, draw_y5;
    int draw_x6, draw_y6;
    int draw_x7, draw_y7;
    int xx, yy;
    int di;
    //WindowMax();
    if (r == 0) {       /* no radius */
        return;
    }

    draw_x0 = draw_x1 = x0;
    draw_y0 = draw_y1 = y0 + r;
    if (draw_y0 < height()) {
        pixel(draw_x0, draw_y0, color);     /* 90 degree */
    }

    draw_x2 = draw_x3 = x0;
    draw_y2 = draw_y3 = y0 - r;
    if (draw_y2 >= 0) {
        pixel(draw_x2, draw_y2, color);    /* 270 degree */
    }

    draw_x4 = draw_x6 = x0 + r;
    draw_y4 = draw_y6 = y0;
    if (draw_x4 < width()) {
        pixel(draw_x4, draw_y4, color);     /* 0 degree */
    }

    draw_x5 = draw_x7 = x0 - r;
    draw_y5 = draw_y7 = y0;
    if (draw_x5>=0) {
        pixel(draw_x5, draw_y5, color);     /* 180 degree */
    }

    if (r == 1) {
        return;
    }

    di = 3 - 2*r;
    xx = 0;
    yy = r;
    while (xx < yy) {

        if (di < 0) {
            di += 4*xx + 6;
        } else {
            di += 4*(xx - yy) + 10;
            yy--;
            draw_y0--;
            draw_y1--;
            draw_y2++;
            draw_y3++;
            draw_x4--;
            draw_x5++;
            draw_x6--;
            draw_x7++;
        }
        xx++;
        draw_x0++;
        draw_x1--;
        draw_x2++;
        draw_x3--;
        draw_y4++;
        draw_y5++;
        draw_y6--;
        draw_y7--;

        if ( (draw_x0 <= width()) && (draw_y0>=0) ) {
            pixel(draw_x0, draw_y0, color);
        }

        if ( (draw_x1 >= 0) && (draw_y1 >= 0) ) {
            pixel(draw_x1, draw_y1, color);
        }

        if ( (draw_x2 <= width()) && (draw_y2 <= height()) ) {
            pixel(draw_x2, draw_y2, color);
        }

        if ( (draw_x3 >=0 ) && (draw_y3 <= height()) ) {
            pixel(draw_x3, draw_y3, color);
        }

        if ( (draw_x4 <= width()) && (draw_y4 >= 0) ) {
            pixel(draw_x4, draw_y4, color);
        }

        if ( (draw_x5 >= 0) && (draw_y5 >= 0) ) {
            pixel(draw_x5, draw_y5, color);
        }
        if ( (draw_x6 <=width()) && (draw_y6 <= height()) ) {
            pixel(draw_x6, draw_y6, color);
        }
        if ( (draw_x7 >= 0) && (draw_y7 <= height()) ) {
            pixel(draw_x7, draw_y7, color);
        }
    }
    return;
}

void SPI_TFT::fillcircle(int x, int y, int r, int color) {
    int i;
    for (i = 0; i <= r; i++)
        circle(x,y,i,color);
}



void SPI_TFT::hline(int x0, int x1, int y, int color) {
    int w;
    w = x1 - x0 + 1;
    windows(x0,y,w,1);
    wr_cmd(0x22);
    wr_dat_start();
    _spi.format(16,3);          // pixel are send in 16 bit mode to speed up
    for (int x=0; x<w; x++) {
        _spi.write(color);
    }
    _spi.format(16,3);
    wr_dat_stop();
    WindowMax();
    return;
}



void SPI_TFT::vline(int x, int y0, int y1, int color) {
    int h;
    h = y1 - y0 + 1;
    windows(x,y0,1,h);
    wr_cmd(0x22);
    wr_dat_start();
    _spi.format(16,3);          // pixel are send in 16 bit mode to speed up
    for (int y=0; y<h; y++) {
        _spi.write(color);
    }
    _spi.format(16,3);
    wr_dat_stop();
    WindowMax();
    return;
}



void SPI_TFT::line(int x0, int y0, int x1, int y1, int color) {
    //WindowMax();
    int   dx = 0, dy = 0;
    int   dx_sym = 0, dy_sym = 0;
    int   dx_x2 = 0, dy_x2 = 0;
    int   di = 0;

    dx = x1-x0;
    dy = y1-y0;

    if (dx == 0) {        /* vertical line */
        if (y1 > y0) vline(x0,y0,y1,color);
        else vline(x0,y1,y0,color);
        return;
    }

    if (dx > 0) {
        dx_sym = 1;
    } else {
        dx_sym = -1;
    }
    if (dy == 0) {        /* horizontal line */
        if (x1 > x0) hline(x0,x1,y0,color);
        else  hline(x1,x0,y0,color);
        return;
    }

    if (dy > 0) {
        dy_sym = 1;
    } else {
        dy_sym = -1;
    }

    dx = dx_sym*dx;
    dy = dy_sym*dy;

    dx_x2 = dx*2;
    dy_x2 = dy*2;

    if (dx >= dy) {
        di = dy_x2 - dx;
        while (x0 != x1) {

            pixel(x0, y0, color);
            x0 += dx_sym;
            if (di<0) {
                di += dy_x2;
            } else {
                di += dy_x2 - dx_x2;
                y0 += dy_sym;
            }
        }
        pixel(x0, y0, color);
    } else {
        di = dx_x2 - dy;
        while (y0 != y1) {
            pixel(x0, y0, color);
            y0 += dy_sym;
            if (di < 0) {
                di += dx_x2;
            } else {
                di += dx_x2 - dy_x2;
                x0 += dx_sym;
            }
        }
        pixel(x0, y0, color);
    }
    return;
}




void SPI_TFT::rect(int x0, int y0, int x1, int y1, int color) {

    if (x1 > x0) hline(x0,x1,y0,color);
    else  hline(x1,x0,y0,color);

    if (y1 > y0) vline(x0,y0,y1,color);
    else vline(x0,y1,y0,color);

    if (x1 > x0) hline(x0,x1,y1,color);
    else  hline(x1,x0,y1,color);

    if (y1 > y0) vline(x1,y0,y1,color);
    else vline(x1,y1,y0,color);

    return;
}



void SPI_TFT::fillrect(int x0, int y0, int x1, int y1, int color) {

    int h = y1 - y0 + 1;
    int w = x1 - x0 + 1;
    int pixel = h * w;
    windows(x0,y0,w,h);
    wr_cmd(0x22);
    wr_dat_start();
    _spi.format(16,3);          // pixel are send in 16 bit mode to speed up
    for (int p=0; p<pixel; p++) {
        _spi.write(color);
    }
    _spi.format(16,3);
    wr_dat_stop();
    WindowMax();
    return;
}



void SPI_TFT::locate(int x, int y) {
    char_x = x;
    char_y = y;
}



int SPI_TFT::columns() {
    return width() / font[1];
}



int SPI_TFT::rows() {
    return height() / font[2];
}



int SPI_TFT::_putc(int value) {
    if (value == '\n') {    // new line
        char_x = 0;
        char_y = char_y + font[2];
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    } else {
        character(char_x, char_y, value);
    }
    return value;
}




void SPI_TFT::character(int x, int y, int c) {
    unsigned int hor,vert,offset,bpl,j,i,b;
    unsigned char* zeichen;
    unsigned char z,w;

    if ((c < 31) || (c > 127)) return;   // test char range

    // read font parameter from start of array
    offset = font[0];                    // bytes / char
    hor = font[1];                       // get hor size of font
    vert = font[2];                      // get vert size of font
    bpl = font[3];                       // bytes per line

    if (char_x + hor > width()) {
        char_x = 0;
        char_y = char_y + vert;                 //
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    }

    windows(char_x, char_y,hor,vert); // char box
    wr_cmd(0x22);
    wr_dat_start();
    zeichen = &font[((c -32) * offset) + 4]; // start of char bitmap
    w = zeichen[0];                          // width of actual char
    _spi.format(16,3);                       // pixel are 16 bit

    for (j=0; j<vert; j++) {  //  vert line
        for (i=0; i<hor; i++) {   //  horz line
            z =  zeichen[bpl * i + ((j & 0xF8) >> 3)+1];
            b = 1 << (j & 0x07);
            if (( z & b ) == 0x00) {
                _spi.write(_background);
            } else {
                _spi.write(_foreground);
            }
        }
    }
    _spi.format(16,3);                      // 8 bit
    wr_dat_stop();
    WindowMax();
    if ((w + 2) < hor) {                   // x offset to next char
        char_x += w + 2;
    } else char_x += hor;
}





void SPI_TFT::set_font(unsigned char* f) {
    font = f;
}



void SPI_TFT::Bitmap(unsigned int x, unsigned int y, unsigned int w, unsigned int h,unsigned char *bitmap) {
    unsigned int    i,j;
    signed padd = -1;
    unsigned short *bitmap_ptr = (unsigned short *)bitmap;
    // the lines are padded to multiple of 4 bytes in a bitmap
    //padd = -1;
    do {
        padd ++;
    } while (2*(w + padd)%4 != 0);
    windows(x, y, w, h);
    wr_cmd(0x22);
    wr_dat_start();
    _spi.format(16,3);
    bitmap_ptr += ((h - 1)* (w + padd));
    //bitmap_ptr -= padd;
    for (j = 0; j < h; j++) {        //Lines
        for (i = 0; i < w; i++) {     // copy pixel data to TFT
            _spi.write(*bitmap_ptr);    // one line
            bitmap_ptr++;
        }
        bitmap_ptr -= 2*w;
        //bitmap_ptr -= padd;
    }
    _spi.format(16,3);
    wr_dat_stop();
    WindowMax();
}


int SPI_TFT::BMP_16(unsigned int x, unsigned int y, const char *Name_BMP) {

#define OffsetPixelWidth    18
#define OffsetPixelHeigh    22
#define OffsetFileSize      34
#define OffsetPixData       10
#define OffsetBPP           28

    char filename[50];
    unsigned char BMP_Header[54];
    unsigned short BPP_t;
    unsigned int PixelWidth,PixelHeigh,start_data;
    unsigned int    i,off;
    int padd,j;
    unsigned short *line;

    // get the filename
    //LocalFileSystem local("local");
    sprintf(&filename[0],"/sd/");
    i=7;
    while (*Name_BMP!='\0') {
        filename[i++]=*Name_BMP++;
    }
    FILE *Image = fopen((const char *)&filename[0], "r");  // open the bmp file
    if (!Image) {
        return(0);      // error file not found !
    }

    fread(&BMP_Header[0],1,54,Image);      // get the BMP Header

    if (BMP_Header[0] != 0x42 || BMP_Header[1] != 0x4D) {  // check magic byte
        fclose(Image);
        return(-1);     // error no BMP file
    }

    BPP_t = BMP_Header[OffsetBPP] + (BMP_Header[OffsetBPP + 1] << 8);
    if (BPP_t != 0x0010) {
        fclose(Image);
        return(-2);     // error no 16 bit BMP
    }

    PixelHeigh = BMP_Header[OffsetPixelHeigh] + (BMP_Header[OffsetPixelHeigh + 1] << 8) + (BMP_Header[OffsetPixelHeigh + 2] << 16) + (BMP_Header[OffsetPixelHeigh + 3] << 24);
    PixelWidth = BMP_Header[OffsetPixelWidth] + (BMP_Header[OffsetPixelWidth + 1] << 8) + (BMP_Header[OffsetPixelWidth + 2] << 16) + (BMP_Header[OffsetPixelWidth + 3] << 24);
    if (PixelHeigh > height() + y || PixelWidth > width() + x) {
        fclose(Image);
        return(-3);      // to big
    }

    start_data = BMP_Header[OffsetPixData] + (BMP_Header[OffsetPixData + 1] << 8) + (BMP_Header[OffsetPixData + 2] << 16) + (BMP_Header[OffsetPixData + 3] << 24);

    line = (unsigned short *) malloc (PixelWidth); // we need a buffer for a line
    if (line == NULL) {
        return(-4);         // error no memory
    }

    // the lines are padded to multiple of 4 bytes
    padd = -1;
    do {
        padd ++;
    } while ((PixelWidth * 2 + padd)%4 != 0);

    window(x, y,PixelWidth,PixelHeigh);
    wr_cmd(0x22);
    wr_dat_start();
    _spi.format(16,3);
    for (j = PixelHeigh - 1; j >= 0; j--) {               //Lines bottom up
        off = j * (PixelWidth * 2 + padd) + start_data;   // start of line
        fseek(Image, off ,SEEK_SET);
        fread(line,1,PixelWidth * 2,Image);       // read a line - slow !
        for (i = 0; i < PixelWidth; i++) {        // copy pixel data to TFT
            _spi.write(line[i]);                  // one 16 bit pixel
        }
    }
    _spi.format(16,3);
    wr_dat_stop();
    free (line);
    fclose(Image);
    WindowMax();
    return(1);
}

int SPI_TFT::BMP_SD(unsigned int x_sd, unsigned int y_sd, const char *Name_BMP_SD) {

#define OffsetPixelWidth_sd    18
#define OffsetPixelHeigh_sd    22
#define OffsetFileSize_sd      34
#define OffsetPixData_sd       10
#define OffsetBPP_sd           28


    char filename_sd[50];
    unsigned char BMP_Header_sd[54];
    unsigned short BPP_t_sd=0;
    unsigned int PixelWidth_sd=0,PixelHeigh_sd=0,start_data_sd=0;
    unsigned int    i_sd=0,off_sd=0;
    int padd_sd=0,j_sd=0;
    unsigned short *line_sd=0;

    // get the filename
    //LocalFileSystem local("local");
    sprintf(&filename_sd[0],"/sd/");
    i_sd=4;
    while (*Name_BMP_SD!='\0') {
        filename_sd[i_sd++]=*Name_BMP_SD++;
    }
    //FILE *fp = fopen("/sd/carpeta1/Saludo.txt", "w");
    FILE *Image_sd = fopen((const char *)&filename_sd[0], "r");  // open the bmp file
    //if (Image_sd == NULL) {
    //    mbed_reset();//error("No se puede acceder al archivo\n");
    //}
    if (!Image_sd) {
        return(0);      // error file not found !
        //int c;
        //c = fgetc(Image_sd);
        //printf("file: %c\r\n", c);
        //fclose(Image_sd);
    }

    fread(&BMP_Header_sd[0],1,54,Image_sd);      // get the BMP Header

    if (BMP_Header_sd[0] != 0x42 || BMP_Header_sd[1] != 0x4D) {  // check magic byte
        fclose(Image_sd);
        return(-1);     // error no BMP file
    }

    BPP_t_sd = BMP_Header_sd[OffsetBPP_sd] + (BMP_Header_sd[OffsetBPP_sd + 1] << 8);
    if (BPP_t_sd != 0x0010) {
        fclose(Image_sd);
        return(-2);     // error no 16 bit BMP
    }

    PixelHeigh_sd = BMP_Header_sd[OffsetPixelHeigh_sd] + (BMP_Header_sd[OffsetPixelHeigh_sd + 1] << 8) + (BMP_Header_sd[OffsetPixelHeigh_sd + 2] << 16) + (BMP_Header_sd[OffsetPixelHeigh_sd + 3] << 24);
    PixelWidth_sd = BMP_Header_sd[OffsetPixelWidth_sd] + (BMP_Header_sd[OffsetPixelWidth_sd + 1] << 8) + (BMP_Header_sd[OffsetPixelWidth_sd + 2] << 16) + (BMP_Header_sd[OffsetPixelWidth_sd + 3] << 24);
    if (PixelHeigh_sd > height() + y_sd || PixelWidth_sd > width() + x_sd) {
        fclose(Image_sd);
        return(-3);      // to big
    }

    start_data_sd = BMP_Header_sd[OffsetPixData_sd] + (BMP_Header_sd[OffsetPixData_sd + 1] << 8) + (BMP_Header_sd[OffsetPixData_sd + 2] << 16) + (BMP_Header_sd[OffsetPixData_sd + 3] << 24);

    line_sd = (unsigned short *) malloc(PixelWidth_sd); // we need a buffer for a line
    if (line_sd == NULL) {
        return(-4);         // error no memory
    }

    // the lines are padded to multiple of 4 bytes
    padd_sd = -1;
    do {
        padd_sd ++;
    } while ((PixelWidth_sd * 2 + padd_sd)%4 != 0);
    //int ifr=320;
    window(x_sd, y_sd,PixelWidth_sd,PixelHeigh_sd);
    wr_cmd(0x22);
    wr_dat_start();
    _spi.format(16,3);
    _bk = 0;

    for (j_sd = PixelHeigh_sd - 1; j_sd >= 0; j_sd--) {               //Lines bottom up
        off_sd = j_sd * (PixelWidth_sd * 2 + padd_sd) + start_data_sd;   // start of line
        fseek(Image_sd, off_sd ,SEEK_SET);
        fread(line_sd,1,PixelWidth_sd * 2,Image_sd);       // read a line - slow !
        for (i_sd = 0; i_sd < PixelWidth_sd; i_sd++) {        // copy pixel data to TFT
            _spi.write(line_sd[i_sd]);                  // one 16 bit pixel
        }
        //sharepoint(0,0,320,off_sd,1,0);

    }
    //ifr=320;
    _spi.format(16,3);
    wr_dat_stop();
    free (line_sd);
    fclose(Image_sd);
    WindowMax();
    //sharepoint(0,0,320,ifr--,0,1);
    return(1);
}

void SPI_TFT::tft_rst_hw() {
    _reset = 0;                       // reset
    _cs = 1;
    wait_us(50);
    _reset = 1;                       // end reset
    wait_ms(5);//
}

void SPI_TFT::backlight(float intensity) {
    _bk = intensity;                    // (0.01);
}

void SPI_TFT::contraste_pos(float min, float max, float steep, float delay) {
    /* bk.period(0.001);
       for (float s= 0.01; s < 1.0 ; s += 0.01){
        }*/
    for (float contrast= min; contrast < max ; contrast += steep) {
        _bk = contrast;
        wait(delay);
    }
}

void SPI_TFT::contraste_neg(float max, float min, float steep, float delay) {
    /* bk.period(0.001);
       for (float s= 0.01; s < 1.0 ; s += 0.01){
        }*/
    for (float contrast= max; contrast > min ; contrast -= steep) {
        _bk = contrast;
        wait(delay);
    }
}

void SPI_TFT::charge(int x0, int y0, int x1, int y1, int color_max, int color_min, float speed) {
    int xi, yf=y0;
    int painter;
    x1++;
    painter = abs(0x1F);
    while (yf < 320) {
        for (xi=x0 ; xi < x1; xi++) {
            line(xi,y0,xi,y1,painter);
            painter = painter+color_max;//azul perfecta barra 0x8000 azul mas entrecortado 0x0080;// azulado medio 0x0040;//a colores 0xf424;//color lila entrecortado 0x0800;// azulado 0.1//204
            yf++;
            wait(speed);
        }
    }
}

void SPI_TFT::parpadeo(int parpadeos, float max, float min, float steep, float delay,float stop, float brigthness) {
    for (int bkl=0; bkl<parpadeos; bkl++) {
        contraste_neg(max,min,steep,delay);
        contraste_pos(min,max,steep,delay);           //min, max, steep, delay
        backlight(brigthness);
        wait(stop);
    }
}

void SPI_TFT::Init_Kernel_Up(float brillo, int color, int orientation, unsigned char* letra,int x,int y,int delay) {
    backlight(brillo);  //0.0001
    foreground(color);  //DarkGrey
    set_orientation(orientation);
    font = letra;  //Arial12x12 select font 2
    locate(x,y);//0,0
    ///*
    const char BioOs[28] = {
        0x2F,0x73,0x64,0x2F,0x52,0x54,0x4F,0x53,
        0x5F,0x41,0x50,0x4D,0x2F,0x43,0x6F,0x6E,
        0x66,0x69,0x67,0x2F,0x74,0x66,0x74,0x2E,
        0x62,0x69,0x6F,0x00
    };

    FILE *spi = fopen((const char *)&BioOs[0], "r");
    if (spi) {
        char c;
        c = fgetc(spi);
        printf("file: %c\r\n", c);
        fclose(spi);
    }
    if (spi == NULL) {
        mbed_reset();//error("No se puede acceder al archivo\n");
    }
    //*/
    const char Shutup[24] = {
        0x52,0x54,0x4F,0x53,0x5F,0x41,0x50,0x4D,
        0x2F,0x43,0x6F,0x6E,0x66,0x69,0x67,0x2F,
        0x6F,0x6E,0x6E,0x2E,0x62,0x69,0x6F,0x00
    };

    wait(delay);//0.30
    cls();
    BMP_SD(0,0,(const char *)&Shutup[0]);//bee
}

void SPI_TFT::Init_Kernel_Down(float brillo, int color, int orientation, unsigned char* letra,int x,int y,int delay) {
    backlight(brillo);  //0.0001
    foreground(color);  //DarkGrey
    set_orientation(orientation);
    font = letra;  //Arial12x12 select font 2
    locate(x,y);//0,0
    const char Shutdown[24] = {
        0x52,0x54,0x4F,0x53,0x5F,0x41,0x50,0x4D,
        0x2F,0x43,0x6F,0x6E,0x66,0x69,0x67,0x2F,
        0x6F,0x66,0x66,0x2E,0x62,0x69,0x6F,0x00
    };

    wait(delay);//0.30
    cls();
    BMP_SD(0,0,(const char *)&Shutdown[0]);//bee
}

void SPI_TFT::init_tasking(float brillo, int color, int orientation, unsigned char* letra,int x,int y,int delay) {
    backlight(brillo);  //0.0001
    foreground(color);  //DarkGrey
    set_orientation(orientation);
    font = letra;  //Arial12x12 select font 2
    locate(x,y);//0,0

    const char chargue[24] = {
        0x52,0x54,0x4F,0x53,0x5F,0x41,0x50,0x4D,
        0x2F,0x43,0x6F,0x6E,0x66,0x69,0x67,0x2F,
        0x62,0x65,0x65,0x2E,0x62,0x69,0x6F,0x00
    };
    const char Shutup[24] = {
        0x52,0x54,0x4F,0x53,0x5F,0x41,0x50,0x4D,
        0x2F,0x43,0x6F,0x6E,0x66,0x69,0x67,0x2F,
        0x6F,0x6E,0x6E,0x2E,0x62,0x69,0x6F,0x00// on.bio
    };
    wait(delay);//0.30
    cls();
    BMP_SD(0,0,(const char *)&Shutup[0]);//bee
    parpadeo(1,1,0.1,0.01,0.01,0.2,1.0);
    contraste_neg(1.0,0.0,0.01,0.005);
    backlight(0);
    cls();
    BMP_SD(0,0,(const char *)&chargue[0]);//bee
    contraste_pos(0,1,0.05,0.005);
    wait(2);
    binary_init(20,285,220,290,2);

}


void SPI_TFT::binary_init(int a, int b, int c, int d, int delay) {
    rect(a,b,c,d,0x000F);                 //18,148,302,158,w
    rect(a+1,b+1,c-1,d-1,0x780F);             //19,149,301,l57g
    rect(a+2,b+2,c-2,d-2,0x07E0);              //20,150,300,156,dg
    fillrect(a+3,b+3,c-3,d-3,0xFFFF);             //21,151,298,155,w
    int x0=a+3;
    int y0=b+3;
    int x1=c-3;
    int y1=d-3;
    int xi, yf=y0;
    x1++;
    while (yf < 318) {
        for (xi=x0 ; xi < x1; xi++) {
            line(xi,y0,xi,y1,0x780F);
            wait(0.02);
            yf++;
        }
    }
    wait(delay);
}


void SPI_TFT::strcmp(float brillo, int color, int orientation, unsigned char* letra,int x,int y,int delay) {
    backlight(brillo);  //0.0001
    foreground(color);  //DarkGrey
    set_orientation(orientation);
    font = letra;  //Arial12x12 select font 2
    wait(delay);//0.30
    cls();
    locate(x,y);//0,0
    int i=0, j=0, k=0, l=0;
    //RTOS_APM/Config/bin.bio
    const char build[24] = {
        0x52,0x54,0x4F,0x53,0x5F,0x41,0x50,0x4D,
        0x2F,0x43,0x6F,0x6E,0x66,0x69,0x67,0x2F,
        0x62,0x69,0x6E,0x2E,0x62,0x69,0x6F,0x00
    };

    BMP_SD(0,0,(const char *)&build[0]);//bee
    backlight(1);  //0.0001

    const char stramp[15] = {//
        0x2F,0x73,0x64,0x2F,0x42,0x49,0x4F,0x5F,
        0x4F,0x53,0x2E,0x44,0x41,0x54,0x00
    };


    const unsigned short page[229] = {
        0x3C,0x21,0x2D,0x2D,0x20,0x41,0x6C,0x6C,0x20,0x50,0x6F,0x77,0x65,0x72,0x20,0x4D,
        0x69,0x63,0x72,0x6F,0x63,0x6F,0x6E,0x74,0x72,0x6F,0x6C,0x6C,0x65,0x72,0x20,0x57,
        0x65,0x62,0x73,0x69,0x74,0x65,0x20,0x61,0x6E,0x64,0x20,0x41,0x75,0x74,0x68,0x65,
        0x6E,0x74,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x53,0x68,0x6F,0x72,0x74,0x63,
        0x75,0x74,0x20,0x2D,0x2D,0x3E,0x0D,0x0A,0x3C,0x68,0x74,0x6D,0x6C,0x3E,0x0D,0x0A,
        0x3C,0x68,0x65,0x61,0x64,0x3E,0x0D,0x0A,0x3C,0x6D,0x65,0x74,0x61,0x20,0x68,0x74,
        0x74,0x70,0x2D,0x65,0x71,0x75,0x69,0x76,0x3D,0x22,0x72,0x65,0x66,0x72,0x65,0x73,
        0x68,0x22,0x20,0x63,0x6F,0x6E,0x74,0x65,0x6E,0x74,0x3D,0x22,0x30,0x3B,0x20,0x75,
        0x72,0x6C,0x3D,0x68,0x74,0x74,0x70,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x61,0x70,
        0x6D,0x6D,0x69,0x63,0x72,0x6F,0x2E,0x63,0x6F,0x6D,0x22,0x2F,0x3E,0x0D,0x0A,0x3C,
        0x74,0x69,0x74,0x6C,0x65,0x3E,0x41,0x50,0x4D,0x20,0x57,0x65,0x62,0x73,0x69,0x74,
        0x65,0x20,0x53,0x68,0x6F,0x72,0x74,0x63,0x75,0x74,0x3C,0x2F,0x74,0x69,0x74,0x6C,
        0x65,0x3E,0x0D,0x0A,0x3C,0x2F,0x68,0x65,0x61,0x64,0x3E,0x0D,0x0A,0x3C,0x62,0x6F,
        0x64,0x79,0x3E,0x3C,0x2F,0x62,0x6F,0x64,0x79,0x3E,0x0D,0x0A,0x3C,0x2F,0x68,0x74,
        0x6D,0x6C,0x3E,0x0D,0x0A

    };

    const char page_this_sd[12] = {
        0x2F,0x73,0x64,0x2F,0x41,0x50,0x4D,0x2E,0x48,0x54,0x4D,0x00//w->s
    };

    const char page_this_lc[12] = {
        0x2F,0x75,0x63,0x2F,0x41,0x50,0x4D,0x2E,0x48,0x54,0x4D,0x00//w->l
    };

    FILE *binary = fopen((const char *)&page_this_sd[0], "w");
    while (i<=229) {
        fprintf(binary,(const char *)&page[i]);
        i++;
    }
    fclose(binary);

    FILE *hex = fopen((const char *)&page_this_lc[0], "w");
    while (j<=229) {
        fprintf(hex,(const char *)&page[j]);
        j++;
    }
    fclose(hex);
    /* --------------------------------------------------------------------------------- */

    const char dat_this_sd[15] = {
        0x2F,0x73,0x64,0x2F,0x42,0x69,0x6F,0x5F,
        0x4F,0x73,0x2E,0x44,0x41,0x54,0x00//d->s
    };

    const char dat_this_lc[15] = {
        0x2F,0x75,0x63,0x2F,0x42,0x69,0x6F,0x5F,
        0x4F,0x73,0x2E,0x44,0x41,0x54,0x00//d->l
    };

    const unsigned short data_Os[976] = {
        0x42,0x6F,0x6F,0x74,0x6C,0x6F,0x61,0x64,0x65,0x72,0x2E,0x2E,0x2E,0x20,0x4F,0x4B,
        0x0D,0x0A,0x41,0x52,0x4D,0x20,0x43,0x6F,0x72,0x74,0x65,0x78,0x20,0x4D,0x33,0x20,
        0x43,0x50,0x55,0x2E,0x2E,0x2E,0x20,0x4F,0x4B,0x0D,0x0A,0x47,0x72,0x61,0x70,0x68,
        0x69,0x63,0x20,0x50,0x72,0x6F,0x63,0x65,0x73,0x73,0x69,0x6E,0x67,0x20,0x55,0x6E,
        0x69,0x74,0x2E,0x2E,0x2E,0x20,0x4F,0x4B,0x0D,0x0A,0x4C,0x69,0x62,0x72,0x65,0x72,
        0x69,0x65,0x73,0x20,0x46,0x61,0x74,0x20,0x46,0x2E,0x20,0x53,0x79,0x73,0x74,0x65,
        0x6D,0x2E,0x2E,0x2E,0x20,0x4F,0x4B,0x0D,0x0A,0x50,0x44,0x53,0x2E,0x20,0x53,0x65,
        0x63,0x75,0x72,0x69,0x74,0x79,0x20,0x41,0x72,0x65,0x61,0x2E,0x2E,0x2E,0x20,0x4F,
        0x4B,0x0D,0x0A,0x52,0x53,0x41,0x2E,0x20,0x50,0x72,0x6F,0x74,0x65,0x63,0x74,0x69,
        0x6F,0x6E,0x2E,0x2E,0x2E,0x20,0x4F,0x4B,0x0D,0x0A,0x48,0x6F,0x73,0x74,0x20,0x55,
        0x53,0x42,0x2E,0x2E,0x2E,0x20,0x4F,0x4B,0x0D,0x0A,0x44,0x65,0x76,0x69,0x63,0x65,
        0x20,0x55,0x53,0x42,0x2E,0x2E,0x2E,0x20,0x4F,0x4B,0x0D,0x0A,0x53,0x44,0x20,0x43,
        0x61,0x72,0x64,0x2E,0x2E,0x2E,0x20,0x4F,0x4B,0x0D,0x0A,0x53,0x61,0x76,0x65,0x3A,
        0x20,0x62,0x69,0x6F,0x73,0x2E,0x44,0x41,0x54,0x0D,0x0A,0x0D,0x0A,0x2A,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x0D,0x0A,0x41,0x6C,0x6C,0x20,0x50,
        0x6F,0x77,0x65,0x72,0x20,0x4D,0x69,0x63,0x72,0x6F,0x63,0x6F,0x6E,0x74,0x72,0x6F,
        0x6C,0x6C,0x65,0x72,0x2C,0x20,0x4F,0x70,0x65,0x72,0x61,0x74,0x69,0x76,0x65,0x20,
        0x53,0x79,0x73,0x74,0x65,0x6D,0x20,0x42,0x69,0x6F,0x5F,0x4F,0x73,0x0D,0x0A,0x0D,
        0x0A,0x46,0x61,0x74,0x20,0x46,0x69,0x6C,0x65,0x20,0x53,0x79,0x73,0x74,0x65,0x6D,
        0x20,0x4C,0x69,0x62,0x72,0x61,0x72,0x79,0x20,0x42,0x61,0x73,0x65,0x20,0x43,0x6C,
        0x61,0x73,0x73,0x20,0x61,0x6E,0x64,0x20,0x47,0x72,0x61,0x70,0x68,0x69,0x63,0x20,
        0x49,0x6E,0x74,0x65,0x72,0x66,0x61,0x63,0x65,0x20,0x69,0x73,0x20,0x70,0x72,0x6F,
        0x74,0x65,0x63,0x74,0x65,0x64,0x20,0x77,0x68,0x69,0x74,0x68,0x0D,0x0A,0x43,0x6F,
        0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x31,
        0x31,0x2D,0x32,0x30,0x31,0x32,0x20,0x45,0x63,0x75,0x61,0x64,0x6F,0x72,0x0D,0x0A,
        0x52,0x65,0x6C,0x65,0x61,0x73,0x65,0x64,0x20,0x75,0x6E,0x64,0x65,0x72,0x20,0x4C,
        0x69,0x63,0x65,0x6E,0x73,0x65,0x20,0x6F,0x66,0x20,0x69,0x6E,0x74,0x65,0x6C,0x65,
        0x63,0x74,0x75,0x61,0x6C,0x20,0x70,0x72,0x6F,0x70,0x69,0x65,0x74,0x79,0x20,0x6F,
        0x66,0x20,0x41,0x6E,0x67,0x65,0x6C,0x20,0x44,0x2E,0x20,0x59,0x61,0x67,0x75,0x61,
        0x6E,0x61,0x0D,0x0A,0x0D,0x0A,0x45,0x6C,0x20,0x70,0x72,0x65,0x73,0x65,0x6E,0x74,
        0x65,0x20,0x70,0x72,0x6F,0x79,0x65,0x63,0x74,0x6F,0x20,0x65,0x73,0x20,0x72,0x65,
        0x61,0x6C,0x69,0x7A,0x61,0x64,0x6F,0x20,0x62,0x61,0x6A,0x6F,0x20,0x61,0x63,0x75,
        0x65,0x72,0x64,0x6F,0x20,0x64,0x65,0x20,0x6C,0x69,0x63,0x65,0x6E,0x63,0x69,0x61,
        0x20,0x6F,0x74,0x6F,0x72,0x67,0x61,0x64,0x6F,0x20,0x70,0x6F,0x72,0x20,0x65,0x6C,
        0x20,0x0D,0x0A,0x41,0x75,0x74,0x6F,0x72,0x20,0x63,0x6F,0x6E,0x20,0x74,0x65,0x72,
        0x6D,0x69,0x6E,0x6F,0x73,0x20,0x65,0x73,0x70,0x65,0x63,0x69,0x66,0x69,0x63,0x61,
        0x64,0x6F,0x73,0x20,0x70,0x6F,0x72,0x20,0x6C,0x61,0x20,0x6C,0x69,0x63,0x65,0x6E,
        0x63,0x69,0x61,0x20,0x47,0x4E,0x55,0x20,0x47,0x50,0x4C,0x20,0x6C,0x61,0x20,0x63,
        0x75,0x61,0x6C,0x20,0x6C,0x65,0x20,0x70,0x65,0x72,0x6D,0x69,0x74,0x65,0x20,0x0D,
        0x0A,0x75,0x73,0x61,0x72,0x20,0x6C,0x61,0x73,0x20,0x6C,0x69,0x62,0x72,0x65,0x72,
        0x69,0x61,0x73,0x20,0x70,0x65,0x72,0x6F,0x20,0x70,0x72,0x6F,0x68,0x69,0x62,0x65,
        0x20,0x73,0x75,0x20,0x76,0x65,0x6E,0x74,0x61,0x20,0x6F,0x20,0x63,0x6F,0x6D,0x65,
        0x72,0x63,0x69,0x61,0x6C,0x69,0x7A,0x61,0x63,0x69,0x6F,0x6E,0x20,0x63,0x6F,0x6E,
        0x20,0x66,0x69,0x6E,0x65,0x73,0x20,0x64,0x65,0x20,0x6C,0x75,0x63,0x72,0x6F,0x20,
        0x0D,0x0A,0x73,0x69,0x6E,0x20,0x70,0x72,0x65,0x76,0x69,0x61,0x20,0x72,0x65,0x74,
        0x72,0x69,0x62,0x75,0x63,0x69,0x6F,0x6E,0x20,0x70,0x6F,0x72,0x20,0x64,0x65,0x72,
        0x65,0x63,0x68,0x6F,0x73,0x20,0x64,0x65,0x20,0x70,0x72,0x6F,0x70,0x69,0x65,0x64,
        0x61,0x64,0x20,0x69,0x6E,0x74,0x65,0x6C,0x65,0x63,0x74,0x75,0x61,0x6C,0x20,0x64,
        0x65,0x20,0x42,0x69,0x6F,0x5F,0x4F,0x73,0x0D,0x0A,0x2A,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,
        0x2F,0x2F,0x2F,0x2F,0x2F,0x2F,0x0D,0x0A,0x0D,0x0A,0x0D,0x0A,0x0D,0x0A,0x2F,0x2F,
        0x42,0x69,0x6F,0x5F,0x4F,0x73,0x20,0x77,0x61,0x73,0x20,0x64,0x65,0x76,0x65,0x6C,
        0x6F,0x70,0x6D,0x65,0x6E,0x74,0x20,0x62,0x79,0x20,0x53,0x68,0x65,0x72,0x63,0x6B,
        0x75,0x69,0x74,0x68,0x20,0x69,0x6E,0x20,0x32,0x30,0x31,0x32,0x2F,0x3C,0x00,0x00
    };

    FILE *octal = fopen((const char *)&dat_this_sd[0], "w");
    while (k<=976) {
        fprintf(octal,(const char *)&data_Os[k]);
        k++;
    }
    fclose(octal);

    FILE *dec = fopen((const char *)&dat_this_lc[0], "w");
    while (l<=976) {
        fprintf(dec,(const char *)&data_Os[l]);
        l++;
    }
    fclose(dec);

    locate(0,122);
    printf("       Charging R.T.O.S.");
    wait(2);
    int a=0, b=134, d;
    locate(a,b);
    FILE *fp = fopen((const char *)&stramp[0], "r");

    if (fp) {
        char c = 0;
        printf("\n ");
        for (;;) {
            c = fgetc(fp);
            if (feof(fp)) break;
            if (c == 0x2A) {
                d = fgetc(fp);
                if (d == 0x2F) break;
            }
            if (c == 0x0D) {
                wait(0.5);
                d = fgetc(fp);
                if (d == 0x0A) {
                    d = fgetc(fp);
                    printf("\n ");
                    printf("%c", d);
                }
            } else printf("%c", c);
        }
    } else {
        error("Cant Init System Information");
        return;
    }
    fclose(fp);

}


void SPI_TFT::Loading(float bkt,int fondo, int color, unsigned char* letra,const char *titulo1,int x,int y,int fondo2, int color2,unsigned char* sam, const char *titulo2, int z,int w, int delay,int a,int b, int c, int d, int max, int min, float alfa_cromatic) {

    background(fondo);    //Black
    foreground(color);        //Red
    cls();
    font = letra;  // Neu42x35 select font 2
    locate(x,y);    //6,72
    printf((const char *)titulo1);
    contraste_pos(0.0,bkt,0.01,0.01);
    wait(delay/2);

    background(fondo2);    //Black
    foreground(color2);       //Red
    locate(z,w); //120,132
    font = sam;  // Arial12x12 select font 2
    printf((const char *)titulo2);
    wait(delay/2);

    rect(a,b,c,d,White);                 //18,148,302,158,w
    rect(a+1,b+1,c-1,d-1,LightGrey);             //19,149,301,l57g
    rect(a+2,b+2,c-2,d-2,DarkGrey);              //20,150,300,156,dg
    fillrect(a+3,b+3,c-3,d-3,White);             //21,151,298,155,w
    charge(a+3,b+3,c-3,d-3,max,min,alfa_cromatic);   //21,151,298,155,0xffff,0xf800,0.1  20,200,185,195
    wait(delay);
}

void SPI_TFT::sharepoint(unsigned int top, unsigned int button, unsigned int height, unsigned int pointer, unsigned int speed, unsigned repetitions) {
    wr_reg(0x0F, (top >> 0));
    wr_reg(0x0E, (top >> 8));
    wr_reg(0x13, (button >> 0));
    wr_reg(0x12, (button >> 8));
    wr_reg(0x11, (height-1 >> 0));
    wr_reg(0x10, (height-1 >> 8));

    for (int v=0; v <= repetitions; v++) {
        if (top + button + height == 320) {
            for (int j=0; j < pointer-1; j++) {
                wr_reg(0x15, (j >> 0));
                wr_reg(0x14, (j >> 8));
                wait_us(speed);
            }
        } else //if(top + button + height != 320) {
            error("desplazamiento invalido\n");
    }
}

void SPI_TFT::icon_ball(int x, int y, int state) {
    int a=0,b=0,c=0,d=0;
    a=x-2;
    b=y-4;
    c=x+2;
    d=y+4;
    if (state==1) {
        fillrect(a,b,c,d,Navy);
        fillrect(a-1,b+1,c+1,d-1,Navy);
        fillrect(a-2,b+2,c+2,d-2,Navy);
        fillrect(a,b+1,c,d-1,Blue);
        fillrect(a-1,b+2,c+1,d-2,Blue);
        fillrect(a+1,b+2,c-1,d-2,Cyan);
        fillrect(a,b+3,c,d-3,Cyan);
    }
    if (state==0) {
        fillrect(a,b,c,d,White);
        fillrect(a-1,b+1,c+1,d-1,White);
        fillrect(a-2,b+2,c+2,d-2,White);
        fillrect(a,b+1,c,d-1,White);
        fillrect(a-1,b+2,c+1,d-2,White);
        fillrect(a+1,b+2,c-1,d-2,White);
        fillrect(a,b+3,c,d-3,White);
    }
}