/*
 * sample.hpp - Example of access to Avago ADNS-9500 laser mouse sensors
 *
 *   Copyright 2012 Jesus Torres <jmtorres@ull.es>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <mbed.h>
#include <stdint.h>

#include "adns9500.hpp"

#define USE_MOTION_BURST
//#define FRAME_CAPTURE

const char* FIRMWARE_FILENAME = "/local/adns9500.fw";
LocalFileSystem local("local");

Ticker printData;
adns9500::ADNS9500 sensor(p11, p12, p13, p15, adns9500::MAX_SPI_FREQUENCY, p14);

bool motionTriggered = false;
bool printDataTriggered = false;

int motionCallbackCounter = 0;

void printDataCallback()
{
    printDataTriggered = true;
}

void motionCallback()
{
    motionTriggered = true;
    motionCallbackCounter++;
}

int main()
{
    int dataReadCounter = 0;
    float totalMotionDx = 0.0;
    float totalMotionDy = 0.0;

#if defined (USE_MOTION_BURST)
    adns9500::MotionData data;
    sensor.attach(&motionCallback);
#elif defined (FRAME_CAPTURE)
    uint8_t frame[adns9500::NUMBER_OF_PIXELS_PER_FRAME];
#else
    sensor.attach(&motionCallback);
#endif

    sensor.reset();

#if ! defined (FRAME_CAPTURE)
    // Firmware upload
    int crc = sensor.sromDownload(FIRMWARE_FILENAME);
    printf("Firmware CRC: 0x%x (%s)\r\n", crc, FIRMWARE_FILENAME);
#endif

    // Enable laser
    sensor.enableLaser();
    printf("Laser enabled\r\n");

#if ! defined (FRAME_CAPTURE)
    printData.attach_us(&printDataCallback, 500);
#endif

    while(true) {
        if (motionTriggered) {
            motionTriggered = false;
            
#if defined (USE_MOTION_BURST)
            sensor.getMotionData(data);
            totalMotionDx += data.dxMM;
            totalMotionDy += data.dyMM;
#else
            float dx, dy;
            sensor.getMotionDeltaMM(dx, dy);
            totalMotionDx += dx;
            totalMotionDy += dy;
#endif
            dataReadCounter++;
        }

        if (printDataTriggered) {
            printDataTriggered = false;
#if defined (USE_MOTION_BURST)
            printf("Motion burst: %f, %f, quality=%d, average=%f, maximum=%d, minimum=%d, "
                   "shutter=%d, periodo=%d, read=%d, irq=%d\r\n",
                    totalMotionDx, totalMotionDy,
                    data.surfaceQuality, data.averagePixel, data.maximumPixel, data.minimumPixel,
                    data.shutter, data.framePeriod, dataReadCounter,
                    motionCallbackCounter);
#else
            printf("Motion delta: %f, %f, read=%d, irq=%d\r\n",
                totalMotionDx, totalMotionDy, dataReadCounter, motionCallbackCounter);
#endif
        }

#if defined (FRAME_CAPTURE)
        printf("FRAME:%d:", dataReadCounter);
        sensor.captureFrame(frame);
        for(uint8_t *p = frame; p != frame + sizeof(frame); ++p)
            printf("%x", *p);
        printf("\r\n");
        dataReadCounter++;
#endif
    }
}
