#pragma once

/*******************************************************************************
 *                       CUSTOM PARAMETERS DECLARATION
 ******************************************************************************/
#ifndef     CONSTANTS_OF_CAR
#define     MAX_SPEED                  230 // km/h

#define     SIZE_SPEED_VECTOR           10
#define     SIZE_AVG_SPEED               3

#define     ON                          1
#define     OFF                         0

#endif  /*! CONSTANTS_OF_CAR */


/*******************************************************************************
 *                           STRUCTURES DECLARATION
 ******************************************************************************/
#ifndef __MY__STRUCTURES_H__
#define __MY__STRUCTURES_H__

#include <vector>
#include "mbed.h"
#include "rtos.h"
#include "MCP23017.h"
#include "WattBob_TextLCD.h"


/** This structure is used to store the RAW engine values
 *  
 *  @param: engineState         [bool] 
 *          State on the engine, on or off
 *  @param: acceleratorValue    [float]
 *          Last value got by TASK 1. From 0 to 1, % of acceleration
 *  @param: brakeValue          [float]
 *          Last value got by TASK 1. From 0 to 1, % of brake
 */
struct engineRAWValues{
    bool    engineState;
    float   acceleratorValue;
    float   brakeValue;
};

/** This structure is used to store the speed and travelled distance of the car
 *  
 *  @param: *p_speedVector         [std::vector<float>] 
 *          Vector used to store the last speed values computed by the simulator
 *          Speed is in m/s.
 *  @param: sizeOfAvg              [int]
 *          Size of the averaging filter used to compute the displayed speed
 *          value. The average is computed by taking the last sizeOfAvg values
 *          of the vector p_speedVector.
 *  @param: averageSpeed           [float]
 *          Last value computed by task 3. In m/s
 *  @param: distance               [float]
 *          Distance travelled by the car, in m.
 */
struct carStatistics{
    // Speed Vector
    std::vector<float> *p_speedVector;
    int                 sizeOfAvg;
    float               averageSpeed;
    // Distance
    float               distance;
};

/** This structure is used by the MAIL Queue object to store information to be
 *  sent to the computer.
 *
 *  @param: avgSpeed        [float]
 *          Last value of average speed computed
 *  @param: accelerator     [float]
 *          Last RAW value of accelerator. From 0 to 1.
 *  @param: brake           [float]
 *          Last RAW value of brake. From 0 to 1.
 */
struct mailStruct{
    float avgSpeed;
    float accelerator;
    float brake;
};


/** This structure is the main structure used by the program in order not to
 *  use global variables. It is given as argument to every function used
 *  in the different RTOSTimer.
 *
 *  @param: System Inputs           [AnalogIn/DigitalIn]
 *          Set of system inputs used by the system for the assignment.
 *  @param: System Outpus           [AnalogOut/DigitalOut]
 *          Set of system outputs used by the system for the assignment.
 * 
 *  @param: Structures              [engineRAWValues/carStatistics]
 *          Two smaller structures used to organized values used by the
 *          different tasks of the assigment.
 * 
 *  @param: *p_mailQueue
 *          Mail Queue object, from "rtos.h". It uses a mailStruct structure
 *          to store information.
 *  @param: nbElementInQueue        [int]
 *          Number of element in the queue. Necessary in order to dump the mail
 *          by another task and because there is no method to get the number
 *          of element stored in the mail queue.
 *
 *  @param: Mutex                   [Mutex]
 *          Set of three mutex to protect the access of the engineRAWValues
 *          and carStatistics structures, as well as the mail queue object.
 *
 *  @param: *p_PC                   [Serial]
 *          Serial connection to the PC
 *
 *  @param: Screen Variables        [MCP23017/WattBob_TextLCD]
 *          Used to access the screen of the MBED shield.
 */
struct carStructure{
    //  System Inputs 
    AnalogIn            *p_accelerator;
    AnalogIn            *p_brake;
    DigitalIn           *p_engineSwitch;
    DigitalIn           *p_sideLightSwitch;
    DigitalIn           *p_leftSwitch;
    DigitalIn           *p_rightSwitch;
    
    //  System Outputs
    DigitalOut          *p_sideLightIndicator;
    PwmOut              *p_leftLight;
    PwmOut              *p_rightLight;
    DigitalOut          *p_engineLight;
    DigitalOut          *p_overspeedLight;
    DigitalOut          *p_brakeLight;
    
    // RAW Values Structure;
    engineRAWValues     *p_rawValues;
    
    // Car Statistics
    carStatistics       *p_carStats;
    
    // Mail Structure
    Mail<mailStruct,100> *p_mailQueue;
    int                 nbElementInQueue;
    
    // Mutex
    Mutex               *p_rawMutex;
    Mutex               *p_statMutex;
    Mutex               *p_mailMutex;
    
    // PC Serial Connection
    Serial              *p_PC;
    
    // Screen variables
    MCP23017            *p_par_port;
    WattBob_TextLCD     *p_lcd;
};

/** Initialize a 'engineRAWValues' structure to zero.
 * 
 *  @param: &structVal [engineRAWValues]
 *          Reference to a structure to set all of its values to 0.
 */
void initRAWValuesStruct(engineRAWValues &structVal);


/** Initialize a 'carStatistics' structure to zero and add the given pointer
 *  to a vector to it
 *
 *  @param: &structStat [carStatistics]
 *          Reference to a structure whose values are set to 0
 *  @param: *speedVec   [std::vector]
 *          Pointer to a vector which is linked to the structure.
 *          Vector used to store the different speed value
 */
void initCarStatisticsStruct(carStatistics &structStat,
                             std::vector<float> *speedVec
                            );

/** Initialize a 'mailStruct' structure to zero.
 * 
 *  @param: &structMail [mailStruct]
 *          Reference to a structure to set all of its values to 0.
 */
void initMailStructure(mailStruct &structMail);

#endif /*! __MY__STRUCTURES_H__ */
