#include "mcp342x.h"

#define LEN_ONE_BYTE 1

MCP342x::MCP342x(I2C *i2c, uint8_t device_address)
{
    _i2c = i2c;

    // The address byte is the device code (4 bits, hardcoded in
    // factory) and the device address (3 bits). These are shifted one
    // bit to the left because mbed uses 8-bit addresses.
    _address = (_device_code << 4) | (device_address << 1);

    // Initialise to default settings: channel 1, gain 1x, 12 bits.
    // It is necessary to do this to ensure that the variables
    // _resolution and _pga are properly set.
    _configuration = 0x10;
    set_channel(CHANNEL_1);
    set_resolution(RESOLUTION_12);
    set_pga(PGA_1);
}

void MCP342x::set_channel(mcp342x_channel_t channel)
{
    _configuration &= REG_CHANNEL_Clear;
    _configuration |= channel << REG_CHANNEL_Pos;
    _write_configuration();
}

void MCP342x::set_conversion_mode(mcp342x_conversion_mode_t mode)
{
    _configuration &= REG_MODE_Clear;
    _configuration |= mode << REG_MODE_Pos;
    _write_configuration();
}

void MCP342x::set_resolution(mcp342x_resolution_t resolution)
{
    _resolution = resolution;

    // _lsb and _max_code are variables required for converting the ADC
    // data into volts; see Section 4.9 of the MCP342x datasheet. Their
    // value depends on the resolution chosen, so it is useful to
    // calculate these values here, whenever the resolution setting is
    // changed.
    //
    // _lsb is the magnitude (in volts) of the last significant byte,
    // and it is calculated as 2 * 2.048 / (2^N), where N is the
    // resolution (datasheet Eq. 4-3).
    //
    // _max_code is the maximum output code, and it is equal to
    // 2^(N-1) - 1 (datasheet Table 4-3).
    switch(_resolution){
        case RESOLUTION_12:
            _lsb = 2 * 2.048 / 4096;
            _max_code = 2047;
            break;
        case RESOLUTION_14:
            _lsb = 2 * 2.048 / 16384;
            _max_code = 8191;
            break;
        case RESOLUTION_16:
            _lsb = 2 * 2.048 / 65536;
            _max_code = 32767;
            break;
        case RESOLUTION_18:
            _lsb = 2 * 2.048 / 262144;
            _max_code = 131071;
            break;
    }

    _configuration &= REG_RESOLUTION_Clear;
    _configuration |= _resolution << REG_RESOLUTION_Pos;
    _write_configuration();
}

void MCP342x::set_pga(mcp342x_pga_t pga)
{
    // The gain value (1, 2, 4 or 8) is required for converting digital
    // output codes to voltage. For this purpose the actual PGA value is
    // kept in the variable _pga, instead of keeping the *position* in
    // the register of PGA, which is what the variable type
    // mcp342x_pga_t  represents.
    switch (pga) {
        case PGA_1:
            _pga = 1;
            break;
        case PGA_2:
            _pga = 2;
            break;
        case PGA_4:
            _pga = 4;
            break;
        case PGA_8:
            _pga = 8;
            break;
    }

    _configuration &= REG_PGA_Clear;
    _configuration |= pga << REG_PGA_Pos;
    _write_configuration();
}

void MCP342x::_write_configuration()
{
    _i2c_command[0] = _configuration;
    _i2c->write(_address, _i2c_command, LEN_ONE_BYTE);
}

uint32_t MCP342x::read()
{
    uint32_t adc_value = 0;
    _i2c->read(_address, _i2c_command, COMMAND_N_BYTES);

    switch (_resolution) {
        case RESOLUTION_12:
            adc_value = (_i2c_command[0] << 8) | _i2c_command[1];
            adc_value &= 0xfff;
            break;

        case RESOLUTION_14:
            adc_value = (_i2c_command[0] << 8) | _i2c_command[1];
            adc_value &= 0x3fff;
            break;

        case RESOLUTION_16:
            adc_value = (_i2c_command[0] << 8) | _i2c_command[1];
            adc_value &= 0xffff;
            break;

        case RESOLUTION_18:
            adc_value = (_i2c_command[0] << 16) |
                (_i2c_command[1] << 8) | _i2c_command[2];
            adc_value &= 0x3ffff;
            break;
    }
    return adc_value;
}

float MCP342x::read_volts(){
    float volts = 0.0;
    uint32_t adc_value = read();

    // The digital output of the MCP342x is in two's complement format;
    // see datasheet Section 4.9. This 'if... else' construction
    // determines whether the digital code is negative or positive; if
    // it is the former, its two's complement is calculated.
    if (adc_value > _max_code) {
        // if the output code is negative...
        volts = (~adc_value & _max_code) + 1;
        volts *= -1;
    } else {
        // if the output code is positive...
        volts = (float)adc_value;
    }

    // The actual voltage is proportional to the resolution and PGA
    // settings. This equation corresponds to Equation 4-4 in the
    // datasheet. The variables _lsb and _pga are calculated whenever
    // the user changes the resolution or PGA parameters.
    return volts * _lsb / _pga;
}
