#include "MCP23008_I2C.hpp"

namespace {
//const uint8_t MCP23008_ADDRESS  = 0x40;

/* MCP23008 registers */
const uint8_t IODIR             = 0x00;
const uint8_t IPOL              = 0x01;
const uint8_t GPINTEN           = 0x02;
const uint8_t DEFVAL            = 0x03;
const uint8_t INTCON            = 0x04;
const uint8_t IOCON             = 0x05;
const uint8_t GPPU              = 0x06;
const uint8_t INTF              = 0x07;
const uint8_t INTCAP            = 0x08;
const uint8_t GPIO              = 0x09;
const uint8_t OLAT              = 0x0A;
};

MCP23008_I2C::MCP23008_I2C ( I2C *i2c, char deviceAddress )
        : _i2c(i2c) {
    _slaveAddress = deviceAddress & 0xFE;

  // Setup the I2C bus
  // The max bitrate for PCF8574 is 100kbit, the max bitrate for MCP23008 is 400kbit, 
  _i2c->frequency(100000);
    reset ();
}

void MCP23008_I2C::set_input_pins ( uint8_t pins ) {
    uint8_t value = read_register ( IODIR );
    write_register ( IODIR, value | pins );
}

void MCP23008_I2C::set_output_pins ( uint8_t pins ) {
    uint8_t value = read_register ( IODIR );
    write_register ( IODIR, value & ~pins );
}

void MCP23008_I2C::write_outputs ( uint8_t values ) {
    write_register ( GPIO, values );
}

uint8_t MCP23008_I2C::read_outputs () {
    return read_register ( OLAT );
}

uint8_t MCP23008_I2C::read_inputs () {
    return read_register ( GPIO );
}

void MCP23008_I2C::set_input_polarity ( uint8_t values ) {
    write_register ( IPOL, values );
}

uint8_t MCP23008_I2C::get_input_polarity () {
    return read_register ( IPOL );
}

void MCP23008_I2C::set_pullups ( uint8_t values ) {
    write_register ( GPPU, values );
}

uint8_t MCP23008_I2C::get_pullups () {
    return read_register ( GPPU );
}

void MCP23008_I2C::interrupt_on_changes ( uint8_t pins ) {
    uint8_t value = read_register ( INTCON );
    value &= ~pins;
    write_register ( INTCON, value );
    value = read_register ( GPINTEN );
    value |= pins;
    write_register ( GPINTEN, value );
}

void MCP23008_I2C::disable_interrupts ( uint8_t pins ) {
    uint8_t value = read_register ( GPINTEN );
    value &= ~pins;
    write_register ( GPINTEN, value );
}

void MCP23008_I2C::acknowledge_interrupt ( uint8_t &pin, uint8_t &values ) {
    pin = read_register ( INTF );
    values = read_register ( INTCAP );
}

uint8_t MCP23008_I2C::read_register ( uint8_t reg ) {
    char data[] = {reg};
    if ( 0 != _i2c->write ( _slaveAddress, data, 1 ) )
        error ( "MCP23008::read_register: Missing ACK for write\n" );

    if ( 0 != _i2c->read ( _slaveAddress, data, 1 ) )
        error ( "MCP23008:read_register: Missing ACK for read\n" );

    return data[0];
}

void MCP23008_I2C::write_register (int reg, int value) {
  char data[] = {reg, value};
    
  _i2c->write(_slaveAddress, data, 2); 
}

void MCP23008_I2C::write_mask ( uint8_t reg, uint8_t mask, bool value ) {
    uint8_t val;
    val = read_register ( reg );
    if ( value )
        val |= mask;
    else
        val &= ~mask;

    write_register ( reg, val );
}

void MCP23008_I2C::reset ( ) {
    write_register ( IODIR, 0xFF );
    for ( uint8_t reg = IPOL; reg <= OLAT; reg++ )
        write_register ( reg, 0 );
}
