#include "mbed.h"

DigitalOut led1(LED1);

#define ROW_COUNT 7
#define LINE_COUNT 5

#define PIN_SCL P0_28
#define PIN_SDA P0_27
#define ADDRESSE_CAP1188 (0x29<<1)

// Some registers we use
#define CAP1188_SENINPUTSTATUS 0x3
#define CAP1188_MTBLK 0x2A
#define CAP1188_LEDLINK 0x72
#define CAP1188_PRODID 0xFD
#define CAP1188_MANUID 0xFE
#define CAP1188_STANDBYCFG 0x41
#define CAP1188_REV 0xFF
#define CAP1188_MAIN 0x00
#define CAP1188_MAIN_INT 0x01
#define CAP1188_LEDPOL 0x73

DigitalOut row[ROW_COUNT] = {
        DigitalOut(P0_0),
        DigitalOut(P0_1),
        DigitalOut(P0_2),
        DigitalOut(P0_3),
        DigitalOut(P0_4),
        DigitalOut(P0_5),
        DigitalOut(P0_23),
};

DigitalOut line[LINE_COUNT][2] = {
        {DigitalOut(P0_6),DigitalOut(P0_7)},
        {DigitalOut(P0_8),DigitalOut(P0_9)},
        {DigitalOut(P0_10),DigitalOut(P0_11)},
        {DigitalOut(P0_15),DigitalOut(P0_16)},
        {DigitalOut(P0_17),DigitalOut(P0_18)},
};


typedef enum {
    C_BLACK=0x0,
    C_GREEN=0x1,
    C_RED=0x2,
    C_YELOW=C_GREEN|C_RED
} tColor;

tColor matrix[ROW_COUNT][LINE_COUNT]={C_BLACK};

PwmOut buzzer(P2_2);
I2C i2c(PIN_SDA, PIN_SCL);
Ticker ticker;

void setMatrix(tColor c) {
    for(int row=0;row<ROW_COUNT;row++)
        for(int line=0;line<LINE_COUNT;line++)
            matrix[row][line]=c;
}

void writeRegisterCap1188(uint8_t reg, uint8_t value) {
    char cmd[2]={reg,value};
    i2c.write(ADDRESSE_CAP1188, cmd, 2);
}

uint8_t readRegisterCap1188(uint8_t reg) {
    i2c.start();
    i2c.write(ADDRESSE_CAP1188);
    i2c.write(reg);
    i2c.start();
    i2c.write(ADDRESSE_CAP1188|1);
    uint8_t ret=(uint8_t)i2c.read(0);
    i2c.stop();
    return ret;
}

bool testWinDir(int row,int line,int dRow, int dLine) {
    int nbTokenAligned=0;
    tColor player=matrix[row][line];
    while(row>=0 && line>=0 && row<ROW_COUNT && line<LINE_COUNT && matrix[row][line]==player) {
        row-=dRow;
        line-=dLine;
    }
    row+=dRow;
    line+=dLine;
    while (row>=0 && line>=0 && row<ROW_COUNT && line<LINE_COUNT && matrix[row][line]==player) {
        row+=dRow;
        line+=dLine;
        nbTokenAligned++;
    }
    return nbTokenAligned>=4;
}

bool testWin(int row,int line) {
    if(testWinDir(row,line,1,0))
        return true;
    if(testWinDir(row,line,1,1))
        return true;
    if(testWinDir(row,line,0,1))
        return true;
    if(testWinDir(row,line,-1,1))
        return true;
    return false;
}

bool addToken(tColor player,int row,int *line) {
    *line=LINE_COUNT-1;
    if(matrix[row][LINE_COUNT-1]!=C_BLACK)
        return false;//collum full
    while((matrix[row][*line-1]==C_BLACK) && (*line>0))
        (*line)--;
    matrix[row][*line]=player;
    return true;
}

void ledMatrixCallback() {
    static int r = 0;
    for(int l=0;l<LINE_COUNT;l++)
    {
        line[l][0]=0;
        line[l][1]=0;
    }
    row[r]=1;
    row[(r+ROW_COUNT-1)%ROW_COUNT]=0;
    for(int l=0;l<LINE_COUNT;l++)
    {
        line[l][0]=matrix[r][l]&C_GREEN;
        line[l][1]=matrix[r][l]&C_RED;
    }
    r = (r+1) % ROW_COUNT;
}

// main() runs in its own thread in the OS
int main() {
    wait(1); //wait extern device start (capasitiv touche)
    buzzer.period(1/440.f);
    buzzer.write(0.0f);
    ticker.attach(ledMatrixCallback, 0.001);

    char oldTouche;
    bool waitNoTouche;
    tColor player=C_GREEN;
    led1=0;


    writeRegisterCap1188(CAP1188_MTBLK, 0);
    // Have LEDs follow touches
    writeRegisterCap1188(CAP1188_LEDLINK, 0xFF);
    // speed up a bit
    writeRegisterCap1188(CAP1188_STANDBYCFG, 0x30);

    while (true) {
        char touche = readRegisterCap1188(CAP1188_SENINPUTSTATUS);
        // clear INT (clear CAP1188_SENINPUTSTATUS)
        writeRegisterCap1188(CAP1188_MAIN, readRegisterCap1188(CAP1188_MAIN) & ~CAP1188_MAIN_INT);
        if (touche!=oldTouche) {
            if (touche) {
                if (waitNoTouche==false) {
                    int row=0;
                    int nbBit=0;
                    for (int l=0;l<ROW_COUNT;l++) {
                        if (touche&(1<<l)) {
                            nbBit++;
                            row=l;
                        }
                    }
                    if (nbBit==1 && row<7) { //if only 1 touche are enable
                        int line;
                        if (addToken(player,row,&line)) {
                            if (testWin(row,line)) {
                                for(int l=0;l<10;l++) {
                                    setMatrix(player);
                                    wait(0.2);
                                    setMatrix(C_BLACK);
                                    wait(0.2);
                                }
                                player=C_GREEN;
                            }
                            else {
                                player=(tColor)(player^0x3);//change player
                            }
                            waitNoTouche=true;
                        }
                        buzzer.write(0.1f);
                        wait(0.2);
                        buzzer.write(0.f);
                    }
                }
            }
            else {
                waitNoTouche=false;
            }
        }
        oldTouche=touche;
    }
}
