
#include "mbed.h"
#include "stm32746g_discovery_lcd.h"
#include "gauge.h"

Serial pc(USBTX, USBRX, 115200);

Gauge::Gauge(int x, int y, int width, int height, uint32_t bgColor, uint32_t borderWidth)
    : m_x(x), m_y(y), m_width(width), m_height(height), m_bgColor(bgColor), m_borderWidth(borderWidth)
{
    m_delta = m_limHigh - m_limLow;
    m_offset = -m_limLow;
    memset(m_text, 0, 30);
}

void Gauge::setLimit(int low, int high)
{
    m_limHigh = high;
    m_limLow = low;
    m_delta = m_limHigh - m_limLow;
    m_offset = -m_limLow;    
}

int Gauge::strlen(uint8_t *str)
{
    int i = 0;
    while(str[i])
        i++;
    return i;
}

void    Gauge::setColorType(uint8_t type)
{
    m_colorType = type;    
}

uint32_t Gauge::getColor(uint32_t position)
{
    float coef = ((float)m_height) / (float)m_delta;
    
    uint8_t red = 255 - ((position * 255) / m_delta) / coef;
    uint8_t blue = 255 - red;
    uint32_t color = 0;
    
    if (m_colorType == 0)
        color = blue + (red <<16) + (255<<24);
    else
        color = red + (blue <<8) + (255<<24);
    
    return color;
}

void Gauge::drawBorder()
{
    int x = m_x;
    int y = m_y;
    while(y < m_y+m_height)
    {
        BSP_LCD_DrawPixel(m_x, y, m_borderColor);
        BSP_LCD_DrawPixel(m_x + m_width, y, m_borderColor);
        y++;
    }
    while(x < m_x+m_width)
    {
        BSP_LCD_DrawPixel(x, m_y, m_borderColor);
        BSP_LCD_DrawPixel(x, m_y + m_height, m_borderColor);
        x++;
    }    
}

void Gauge::draw()
{
    drawBorder();
    int percentValue = getGaugePercentValue(m_value);
    
    int x = m_x+1;
    while (x < m_x + m_width) {
        int y = m_y +1;
        while (y < m_y + m_height) {
            uint32_t color = 0;
            if (percentValue < 100 - getGaugePercentPosition(y))
                color = LCD_COLOR_LIGHTGRAY;
            else
                color = getColor(y - m_y);
            BSP_LCD_DrawPixel(x, y, color);
            y++;
        }
        x++;
    }
    printValue();
    printRange();
    printText();
}

void Gauge::printValue()
{
    memset(tValue, 0, 6);
    sprintf(tValue, "%d", m_value);
    BSP_LCD_DisplayStringAt((m_x + m_width/2)- 8*(strlen((uint8_t *)tValue)/2), (m_y + m_height) + 5, (uint8_t *)&tValue, LEFT_MODE);
}

void Gauge::printText()
{
    BSP_LCD_DisplayStringAt((m_x + m_width/2)- 8*(strlen((uint8_t *)m_text)/2), m_y  - 15, (uint8_t *)&m_text, LEFT_MODE);
}

void Gauge::printRange()
{
    memset(textValue, 0, 6);

    sprintf(textValue, "%d", m_limLow);
    BSP_LCD_DisplayStringAt(m_x- 8*(strlen((uint8_t *)textValue)), m_y + m_height - 12, (uint8_t *)&textValue, LEFT_MODE);
    sprintf(textValue, "%d", m_limHigh);
    BSP_LCD_DisplayStringAt(m_x- 8*(strlen((uint8_t *)textValue)), m_y, (uint8_t *)&textValue, LEFT_MODE);
    sprintf(textValue, "%d", (m_limHigh + m_limLow)/2);
    BSP_LCD_DisplayStringAt(m_x- 8*(strlen((uint8_t *)textValue)), m_y + m_height/2 - 8, (uint8_t *)&textValue, LEFT_MODE);
}

void Gauge::setValue(int value)
{
    m_value = value;
    printValue();
    draw();
}
void Gauge::setText(const char* str)
{
    sprintf((char*)m_text, str);
}

int8_t Gauge::getGaugePercentValue(int value)
{
//    ((input - min) * 100) / (max - min)
    uint32_t delta = m_limHigh - m_limLow;

    int32_t perc = ((value - m_limLow) * 100) / (delta);
    return perc;
}

int8_t Gauge::getGaugePercentPosition(int value)
{
//    ((input - min) * 100) / (max - min)
    uint32_t delta = (m_height + m_y) - m_y;

    int32_t perc = ((value - m_y) * 100) / (delta);
    return perc;
}

