/**
 * @file    UartRPCFunctions.cpp
 * @brief   BLE UART RPC stubs implementation
 * @author  Doug Anson
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
 #include "UartRPCFunctions.h"
 #include "UartRPC.h"
 
 
 #include "Base64.h"
 
 #ifdef DBG
    #undef DBG
 #endif
 #define DBG  std::printf
 
 static UartRPC *__rpc = NULL;
 
 void ble_rpc_init(BLEDevice &ble) 
 {
     if (__rpc == NULL) __rpc = new UartRPC(ble);
 }
 
 bool ble_rpc_open_udp_socket(char *ip_address,int port,ble_dispatch_callback_fn cb) 
 {
     bool status = false;
     uint8_t response[2];
     memset(response,0,2);
     
     //DBG("ble_rpc_open_udp_socket: UDP Socket open() dispatching...\r\n");
     if (__rpc->dispatch(SOCKET_OPEN_FN,(void *)cb,response,2,"%s %d",ip_address,port) > 0) {
         //DBG("ble_rpc_open_udp_socket: UDP Socket open() dispatched SUCCESSFULLY\r\n");
         status = true;
     }
     
     // failure
     //DBG("ble_rpc_open_udp_socket: UDP Socket open() dispatch failed\r\n");
     return status;
 }
 
 bool ble_rpc_close_udp_socket(void) 
 {
     bool status = false;
     uint8_t response[2];
     memset(response,0,2);
     
     //DBG("ble_rpc_open_udp_socket: UDP Socket close() dispatched successfully. Waiting on response...\r\n");
     if (__rpc->dispatch(SOCKET_CLOSE_FN,NULL,response,2,"%s","loc") > 0) {
         //DBG("ble_rpc_close_udp_socket: UDP Socket close() dispatched SUCCESSFULLY\r\n");
         status = true;
     }
     
     // failure
     //DBG("ble_rpc_close_udp_socket: UDP Socket close() dispatch failed\r\n");
     return status;
 }
 
 int ble_rpc_send_data(uint8_t *data,int data_length) 
 {
     Base64 b64;
     uint8_t response[2];
     memset(response,0,2);
     int base64_data_length = MAX_ARGUMENT_LENGTH;
     //DBG("ble_rpc_send_data: base64 encoding data...\r\n");
     char *base64_data = b64.Encode((char *)data,data_length,(std::size_t *)&base64_data_length);
     //DBG("ble_rpc_send_data: sending data=[%s] length=%d...\r\n",base64_data,strlen((char *)base64_data));
     int sent_length = __rpc->dispatch(SEND_DATA_FN,NULL,response,2,"%s",base64_data);
     //DBG("ble_rpc_send_data: dispatched %d bytes\r\n",sent_length); 
     if (base64_data != NULL) free(base64_data);
     return sent_length;
 }
 
 int ble_rpc_get_location(BLELocation *location) 
 {
     uint8_t response[2];
     memset(response,0,2);
     __rpc->setLocationInstance(location);
     return __rpc->dispatch(GET_LOCATION_FN,NULL,response,2,"%s","");
 }
 
 int ble_rpc_recv_packet(uint8_t *buffer,int buffer_length)
 {  
     int n = 0;
     //DBG("ble_rpc_recv_packet: checking for a local dispatch...\r\n");
     if (__rpc->localDispatchAvailable()) { 
        //DBG("ble_rpc_recv_packet: local dispatch is ready. Retrieving...\r\n");
        n =  __rpc->retrieveLocalDispatch(buffer,buffer_length);
     }
     return n;
 }
 
 int ble_rpc_recv_data(uint8_t *buffer,int buffer_length)
 {
     //DBG("ble_rpc_recv_data: got data...\r\n");
     if (__rpc->accumulate(buffer,buffer_length)) {
         //DBG("ble_rpc_recv_data: received entire packet.. dispatching locally...\r\n");
         __rpc->dispatch();
     }
     return buffer_length;
 } 
 