/**
 * @file    network_stubs.cpp
 * @brief   mbed Endpoint network stubs implementation (BLE)
 * @author  Doug Anson
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#include "network_stubs.h"

#include "UartRPCFunctions.h"

extern "C" {
    
// Globals
BLEDevice ble;
volatile bool __registered = false;

static const uint16_t deviceInfoServiceUUID[] = {GattService::UUID_DEVICE_INFORMATION_SERVICE};

char _ipAddress[IP_ADDRESS_LENGTH];
int  _ipPort;

// record the remote UDPSocket endpoint IP address
void ble_set_remote_ip_address(const char *ipAddress,const int ipAddressLength)
{
    memset(_ipAddress,0,IP_ADDRESS_LENGTH);
    int length = ipAddressLength;
    if (length > IP_ADDRESS_LENGTH) length = IP_ADDRESS_LENGTH;
    if (ipAddress != NULL) memcpy(_ipAddress,ipAddress,length);
}

// record the remote UDPSocket endpoint port
void ble_set_remote_ip_port(const int ipPort)
{
    _ipPort = ipPort;
}

// BLE Disconnection callback
void ble_disconnect_callback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason) 
{
    DBG("Disconnected handle %u!\r\n", handle);
    __registered = false;
    DBG("Restarting the advertising process\r\n");
    ble.startAdvertising();
}

// BLE UDP socket open completion callback
void ble_rpc_completed_open_udp_socket(bool opened) {
    if (opened) {
        // NSP registration
        DBG("ble_connect_callback: calling NSP registration...\r\n");
        register_endpoint(true);
        DBG("ble_connect_callback: NSP registration completed\r\n");
        
        // registration completed
        __registered = true;
    }
    else {
        DBG("ble_connect_callback: remote UDP socket open FAILED...\r\n");
        __registered = false;
    }
}

#if defined(TARGET_STM32F411RE) || defined(TARGET_STM32F401RE)
void ble_connect_callback(Gap::Handle_t handle, const Gap::ConnectionParams_t *reason) 
#else
// BLE Connection callback
void ble_connect_callback(Gap::Handle_t handle, Gap::addr_type_t peerAddrType, const Gap::address_t peerAddr, Gap::addr_type_t ownAddrType, const Gap::address_t ownAddr, const Gap::ConnectionParams_t *params) 
#endif
{
    DBG("ble_connect_callback: BLE connected!...\r\n");    
    
    // 9/4/15: Android 5.x appears to have some sort of poor-mans DDoS detector in its BLE stack so we have to slow things down a bit.. 
 #ifdef ANDROID_BLE_DELAY
    wait_ms(ANDROID_BLE_DELAY_MS);
 #endif
 
    if (__registered == false) {
        DBG("ble_connect_callback: Opening remote UDP socket to: %s@%d...\r\n",_ipAddress,_ipPort);
        ble_rpc_open_udp_socket(_ipAddress,_ipPort,&ble_rpc_completed_open_udp_socket);   
    }
    else {
        DBG("ble_connect_callback: Already registered and connected...\r\n");
    }
}

// BLE Timeout callback
void ble_timeout_callback(void) 
{
    DBG("BLE TIMEOUT OCCURED\r\n");
}

// BLE Init
void ble_init(void) 
{
    // BLE initialization and callback setup
    __registered = false;
    ble.init();
    ble.onDisconnection(ble_disconnect_callback);
    ble.onConnection(ble_connect_callback);
    ble.onTimeout(ble_timeout_callback);
}

// BLE Begin Advertising
void ble_begin_advertising(void) 
{
    extern uint8_t endpoint_name[NODE_NAME_LENGTH];             // our NSP NODE name
    
    DBG("Starting BLE Advertising (%s)...\r\n",endpoint_name);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_REMOTE_CONTROL);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME,(uint8_t *)endpoint_name,strlen((char *)endpoint_name));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,(const uint8_t *)endpoint_name, strlen((char *)endpoint_name));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS,(uint8_t *)deviceInfoServiceUUID, sizeof(deviceInfoServiceUUID));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,(const uint8_t *)UARTServiceUUID, sizeof(UARTServiceUUID));
#if defined(TARGET_STM32F411RE) || defined(TARGET_STM32F401RE)
    ble.setAdvertisingInterval(1000);
#else
    ble.setAdvertisingInterval(Gap::MSEC_TO_ADVERTISEMENT_DURATION_UNITS(1000));  
#endif  
    ble.startAdvertising();
}

// BLE send data
int ble_send_data(uint8_t *data,int data_len)
{
    int sent = 0;
    
    /*
    DBG("in ble_send_data() data_len=%d data: [",data_len);
    for(int i=0;i<data_len;++i) {
        DBG("%.2x",data[i]);
        if (i < (data_len-1)) DBG(" ");
    }
    DBG("] sent: ");
    */
   
    sent = ble_rpc_send_data(data,data_len);   
    //DBG("%d bytes\r\n",sent);
    DBG("ble_send_data(): sent %d bytes...\r\n",sent);
    return sent;
}

// BLE recv data
int ble_recv_data(uint8_t *buffer,int buffer_len)
{
    int recv = ble_rpc_recv_data(buffer,buffer_len);
    
    /*
    DBG("in ble_recv_data() recv=%d data: [",recv);
    for(int i=0;i<recv;++i) {
        DBG("%.2x",buffer[i]);
        if (i < (recv-1)) DBG(" ");
    }
    DBG("] buffer_len=%d\r\n",buffer_len);
    */
    DBG("in ble_recv_data() received %d bytes...\r\n",recv);
    
    return recv;
}
    
// plumb out the network
void net_stubs_pre_plumb_network(bool canActAsRouterNode) 
{
    // BLE initialize...
    DBG("Initializing BLE...\r\n");
    ble_init();
}

// called after the endpoint is configured...
void net_stubs_post_plumb_network(void) 
{
    // Initialize the BLE RPC layer
    DBG("Initializing BLE UART RPC layer...\r\n");
    ble_rpc_init(ble);
    
    // BLE advertise...
    DBG("Beginning BLE advertising...\r\n");
    ble_begin_advertising();
}

// create a suitable main event loop for this specific network
void net_stubs_create_main_loop(void)
{
   // nothing to do for BLE endpoints - we are using the Ticker-based loop in nsdl_support.cpp
   ; 
}

// register the endpoint
void net_stubs_register_endpoint(void)
{
    // not used.. must wait until we get a BLE connect callback.. then register...
    ;
}

// begin the main loop for processing network events
void net_stubs_begin_main_loop(void) 
{
    // NDSL main loop
    DBG("Beginning NSDL Main Event Loop...\r\n");
    nsdl_event_loop();
}

}
