/**
 * @file    Dispatcher.h
 * @brief   BLE UART RPC dispatcher header
 * @author  Doug Anson
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
 #ifndef __RPC_DISPATCH_H__
 #define __RPC_DISPATCH_H__
 
 #include "BLEDevice.h"
 #include "UARTService.h"
 #include "SplitterAssembler.h"
 
 // These are aligned with MAX_VALUE_BUFFER_LENGTH in mbedConnectorInterface.h 
 #define MAX_PACKET_LENGTH          340            // longest packet we will send (aligns with 15x20 in SplitterAssembler)
 #define MAX_ARGUMENT_LENGTH        340            // longest argument sent as parameter list (aligns with 15x20 in SplitterAssembler)
 #define MAX_RESULT_LENGTH          340            // longest result received (aligns with 15x20 in SplitterAssembler)
 
 // This is aligned with the value of DEF_FRAGMENT_LENGTH in SplitterAssembler.h and is also the max UART MTU for BLE
 #define UART_SEGMENT_LENGTH        20             // BLE: max UART MTU (will trigger send immediately...)
  
 class Dispatcher {
     public:
        /**
        Default constructor
        @param ble the BLEDevice instance
        */
        Dispatcher(BLEDevice &ble);
        
        /**
        Dispatch function request with arguments (already serialized and prepared for transport)
        @param fn_id the identifier for the target remote function to invoke
        @param args the serialized function argument array list
        @param args_length the length of the serialized function argument list
        @param response the received response buffer
        @param response_length the recieved response buffer length
        @returns number of bytes sent in dispatch() or -1 if in failure
        */
        int dispatch(uint8_t fn_id,uint8_t *args,int args_length,uint8_t *response,int response_length);
        
     private:
        void                 onDataWritten(const GattCharacteristicWriteCBParams *params);
        
        int                  uart_write(bool send_ack = false);
        int                  uart_write(uint8_t *data,int data_length);
        int                  prepare_send_packet(uint8_t fn_id,uint8_t *args,int args_length);
        
        uint8_t              m_send_packet[MAX_PACKET_LENGTH+1];
        UARTService          m_uart;
        SplitterAssembler    m_splitter;
        int                  m_splitter_count;
        int                  m_splitter_index;
 };
 
 #endif // __RPC_DISPATCH_H__