/**
 * @file    Utils.cpp
 * @brief   mbed CoAP Endpoint misc utils collection
 * @author  Doug Anson
 * @version 1.0
 * @see
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// mbed Endpoint includes
#include "mbed-connector-interface/ConnectorEndpoint.h"
#include "mbed-connector-interface/OptionsBuilder.h"
#include "mbed-connector-interface/mbedEndpointNetwork.h"
#include "mbed-connector-interface/DeviceManager.h"

// Maximum CoAP URL length
#define MAX_CONN_URL_LENGTH		128

// External references (defined in main.cpp)
Connector::Options *configure_endpoint(Connector::OptionsBuilder &builder);
extern Logger logger;

// Our Endpoint
Connector::Endpoint *__endpoint = NULL;

// Our Endpoint configured Options
Connector::OptionsBuilder config;
Connector::Options *options = NULL;

// initialize the Connector::Endpoint instance
void *utils_init_endpoint(bool canActAsRouterNode) {
	// alloc Endpoint
    logger.log("Endpoint: allocating endpoint instance...");
	Connector::Endpoint *ep = new Connector::Endpoint(&logger,options);
	if (ep != NULL) {
		config.setEndpoint((void *)ep);
		ep->asRouterNode(canActAsRouterNode);
    }
    return (void *)ep;
}

// further simplifies the endpoint main() configuration by removing the final initialization details of the endpoint...
void utils_configure_endpoint(void *p)
{   
	// our Endpoint 
	Connector::Endpoint *ep = (Connector::Endpoint *)p;
	
    // NSP/NSDL default configuration - see mbedConnectorInterface.h for definitions... 
    logger.log("Endpoint: setting defaults...");
    config.setEndpointNodename(NODE_NAME);
    config.setEndpointType(NSP_ENDPOINT_TYPE);
    config.setRegUpdatePeriod(REG_UPDATE_PERIOD_MS);
    config.setLifetime(REG_LIFETIME_SEC);

    // WiFi defaults
    config.setWiFiSSID((char *)WIFI_DEFAULT_SSID);          			// default: changeme
    config.setWiFiAuthType(WIFI_NONE);      							// default: none
    config.setWiFiAuthKey((char *)WIFI_DEFAULT_AUTH_KEY);   			// default: changeme
    
    // Default CoAP Connection Type
    config.setCoAPConnectionType(COAP_UDP);								// default CoAP Connection Type
    
    // Set the default IP Address Type
    config.setIPAddressType(IP_ADDRESS_TYPE_IPV4);						// IPv4 is the default
    
    // 802.15.4 defaults
    unsigned char psk[16];
    unsigned char psk_identity[2];
    memset(psk,0,16);
    memset(psk_identity,0,2);
    config.setPreSharedKey(psk);
    config.setPreSharedKeyIdentity(psk_identity);
                 
    // Establish default CoAP observation behavior
    config.setImmedateObservationEnabled(true);    

    // Establish default CoAP GET-based observation control behavior
    config.setEnableGETObservationControl(false);    

	// Device Manager installation
    DeviceManager *device_manager = (DeviceManager *)ep->getDeviceManager();
    if (device_manager != NULL) {
    	logger.log("Endpoint: installing and setting up device manager and its resources...");
    	device_manager->install((void *)ep,(void *)&config);
    }
    else {
    	logger.log("Endpoint: no device manager installed...");
    }
    
    // main.cpp can override or change any of the above defaults...
    logger.log("Endpoint: gathering configuration overrides...");
    options = configure_endpoint(config);
    
    // set our options
	ep->setOptions(options);
	
    // DONE
    logger.log("Endpoint: endpoint configuration completed.");
}

// build out  the endpoint and its resources
void utils_build_endpoint(void *p)
{
    if (p != NULL) {
    	// Build the Endpoint
    	logger.log("Endpoint: building endpoint and its resources...");
		Connector::Endpoint *ep = (Connector::Endpoint *)p;
	    ep->build_endpoint();
	}
}

// parse out the CoAP port number from the connection URL
uint16_t extract_port_from_url(char *url,uint16_t default_port) 
{
	uint16_t port = default_port;
	
	if (url != NULL && strlen(url) > 0) {
		char buffer[MAX_CONN_URL_LENGTH+1];
		char uri[MAX_CONN_URL_LENGTH+1];
		char path[MAX_CONN_URL_LENGTH+1];
		
		// initialize the buffer
		memset(buffer,0,MAX_CONN_URL_LENGTH+1);
		memset(uri,0,MAX_CONN_URL_LENGTH+1);
		memset(path,0,MAX_CONN_URL_LENGTH+1);
		int length = strlen(url); 
		
		// truncate if needed
		if (length >MAX_CONN_URL_LENGTH) length = MAX_CONN_URL_LENGTH;
		
		// make a copy...
		memcpy(buffer,url,length);
		
		// remove the forward slashes and colons
		for(int i=0;i<length;++i) {
			if (buffer[i] == ':') buffer[i] = ' ';
			if (buffer[i] == '/') buffer[i] = ' ';
		}
		
		// parse
		sscanf(buffer,"%s %s %d",uri,path,(int *)&port);
	}
	
	// DEBUG
	logger.log("Endpoint: CoAP port: %u",port);
	
	// return the port
	return port; 
}