/**
 * @file    DeviceFirmwareCompositeResource.cpp
 * @brief   mbed CoAP Endpoint Device Firmware Composited Resource
 * @author  Doug Anson
 * @version 1.0
 * @see
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
// Class 
#include "mbed-connector-interface/DeviceFirmwareCompositeResource.h"

// Endpoint Class
#include "mbed-connector-interface/ConnectorEndpoint.h"
 
// Options Builder
#include "mbed-connector-interface/OptionsBuilder.h"

// DeviceManagementResponder
#include "mbed-connector-interface/DeviceManagementResponder.h"

// DeviceStringResource
#include "mbed-connector-interface/DeviceStringResource.h"

// DeviceIntegerResource
#include "mbed-connector-interface/DeviceIntegerResource.h"

// DeviceFirmwareUpdateResource
#include "mbed-connector-interface/DeviceFirmwareUpdateResource.h"

// DeviceFirmwareManifestResource
#include "mbed-connector-interface/DeviceFirmwareManifestResource.h"
 
// Default Constructor
DeviceFirmwareCompositeResource::DeviceFirmwareCompositeResource(const Logger *logger,const char *obj_name,const void *dm_responder) {
    this->m_logger = (Logger *)logger;
    this->m_dm_responder = (void *)dm_responder;
    this->m_obj_name = (char *)obj_name;
    for(int i=0;i<NUM_FIRMWARE_RESOURCE_IDS;++i) {
        this->m_firmware_resource[i] = NULL;
    }
}
 
// Copy Constructor
DeviceFirmwareCompositeResource::DeviceFirmwareCompositeResource(const DeviceFirmwareCompositeResource &res) {
    this->m_logger = res.m_logger;
    this->m_dm_responder = res.m_dm_responder;
    this->m_obj_name = res.m_obj_name;
    for(int i=0;i<NUM_FIRMWARE_RESOURCE_IDS;++i) {
        this->m_firmware_resource[i] = res.m_firmware_resource[i];
    }
}
 
// Destructor
DeviceFirmwareCompositeResource::~DeviceFirmwareCompositeResource() {
}

// Build out our Firmware Resources
void DeviceFirmwareCompositeResource::buildResources() {
    // LWM2M_PAYLOAD_ID (TODO: Needs to be OPAQUE and full firmware blob...)
    this->m_firmware_resource[LWM2M_PAYLOAD_ID] = new DeviceStringResource(this->m_logger,this->m_obj_name,LWM2M_DEV_FIRMWARE_PACKAGE_ID,"fw_package",this->m_dm_responder,false);
    
    // LWM2M_MANIFEST_ID
    this->m_firmware_resource[LWM2M_MANIFEST_ID] = new DeviceFirmwareManifestResource(this->m_logger,this->m_obj_name,LWM2M_DEV_FIRMWARE_PACKAGE_URI_ID,this->m_dm_responder,false);
    
    // LWM2M_UPDATE_ID
    this->m_firmware_resource[LWM2M_UPDATE_ID] = new DeviceFirmwareUpdateResource(this->m_logger,this->m_obj_name,LWM2M_DEV_FIRMWARE_UPDATE_ID,this->m_dm_responder,false);
    
    // LWM2M_STATE_ID
    this->m_firmware_resource[LWM2M_STATE_ID] = new DeviceIntegerResource(this->m_logger,this->m_obj_name,LWM2M_DEV_FIRMWARE_STATE_ID,"fw_state",this->m_dm_responder,false);
    
    // LWM2M_RESULT_ID
    this->m_firmware_resource[LWM2M_RESULT_ID] = new DeviceIntegerResource(this->m_logger,this->m_obj_name,LWM2M_DEV_FIRMWARE_RESULT_ID,"fw_result",this->m_dm_responder,true);
    
    // LWM2M_PKGNAME_ID
    this->m_firmware_resource[LWM2M_PKGNAME_ID] = new DeviceStringResource(this->m_logger,this->m_obj_name,LWM2M_DEV_FIRMWARE_PKG_NAME_ID,"fw_pkg_name",this->m_dm_responder,false);
    
    // LWM2M_PKGVERSION_ID
    this->m_firmware_resource[LWM2M_PKGVERSION_ID] = new DeviceStringResource(this->m_logger,this->m_obj_name,LWM2M_DEV_FIRMWARE_PKG_VERSION_ID,"fw_pkg_version",this->m_dm_responder,false);
    
    // Establish default values for State and Result
    this->m_firmware_resource[LWM2M_STATE_ID]->put("1");         // idle == 1
    this->m_firmware_resource[LWM2M_RESULT_ID]->put("0");        // default result == 0  
    
    // Tie firmare composite resource for future updating by the dm_responder functions
    DeviceManagementResponder *dmr = (DeviceManagementResponder *)this->m_dm_responder;
    dmr->setFirmwareCompositeResource(this);
}
 
// add resources
void DeviceFirmwareCompositeResource::addResources(void *cfg) {
    // our Endpoint configuration
    Connector::OptionsBuilder *configure_options = (Connector::OptionsBuilder *)cfg;
    
    // loop through the internal resources and add them... 
    for(int i=0;i<NUM_FIRMWARE_RESOURCE_IDS;++i) {
        if (this->m_firmware_resource[i] != NULL) {
            configure_options->addResource(this->m_firmware_resource[i]);
        }
    }
}

// get a specific resource
void *DeviceFirmwareCompositeResource::getResource(FirmwareResourceIDs id) {
    int index = (int)id;
    if (index >= 0 && index < NUM_FIRMWARE_RESOURCE_IDS) {
        return this->m_firmware_resource[index];
    }
    return NULL;
}


 