/**
 * @file    Endpoint.h
 * @brief   mbed CoAP Endpoint base class
 * @author  Doug Anson/Chris Paola
 * @version 1.0
 * @see
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __CONNECTOR_ENDPOINT_H__
#define __CONNECTOR_ENDPOINT_H__

// mbed-client support
#include "mbed-client/m2minterfacefactory.h"
#include "mbed-client/m2mdevice.h"
#include "mbed-client/m2minterfaceobserver.h"
#include "mbed-client/m2minterface.h"
#include "mbed-client/m2mobjectinstance.h"
#include "mbed-client/m2mresource.h"

// Support for Logging/Debug output
#include "mbed-connector-interface/Logger.h"

// Options support
#include "mbed-connector-interface/Options.h"

// Connector namespace
namespace Connector  {

/** Endpoint class
 */
class Endpoint : public M2MInterfaceObserver  {

public:
    /**
    Default Constructor
    */
    Endpoint(const Logger *logger,const Options *ob);

    /**
    Copy Constructor
    @param ob input endpoint instance to deep copy
    */
    Endpoint(const Endpoint &ep);

    /**
    Destructor
    */
    virtual ~Endpoint();

    /**
    Build out the endpoint.
    */
    void build_endpoint();

    /**
    Plumb the lower RF network stack
    @param device_manager input optional device manager (DeviceManager type)
    @param canActAsRouterNode input boolean indicating whether this node can act as a router node or not.
    */
    static void plumbNetwork(void *device_manager = NULL,bool canActAsRouterNode = false);

    /**
    Initialize the endpoint's configuration and begin the endpoint's main even loop
    */
    static void start();
    
    // register the endpoint
	void register_endpoint(M2MSecurity *server_instance, M2MObjectList resources);
	
	// re-register endpoint
	void re_register_endpoint();
	
	// de-register endpoint and stop scheduling
	void de_register_endpoint(void);
	
	//  mbed-client : object registered
    virtual void object_registered(M2MSecurity *server_instance, const M2MServer &server);

	//  mbed-client : registration updated
	virtual void registration_updated(M2MSecurity *server_instance, const M2MServer &server) ;

	//  mbed-client : object unregistered
	virtual void object_unregistered(M2MSecurity *server_instance);

    //  mbed-client : bootstrap done
	virtual void bootstrap_done(M2MSecurity *server_instance);

	//  mbed-client : resource value updated
	virtual void value_updated(M2MBase *resource, M2MBase::BaseType type) ;
	
	//  mbed-client : error handler
	virtual void error(M2MInterface::Error error);
	
	// get our Options
	void setOptions(Options *options);
	
	// get our Options
	Options *getOptions();
	
	// get our Server
	M2MSecurity *getServer();
	
	// get our Object List
	M2MObjectList  getObjectList();
	
	// configure to act as router node
	void asRouterNode(bool canActAsRouterNode);
	
	// access the logger
    Logger *logger();
	
	// set the device manager
    void setDeviceManager(void *device_manager);
    
    // get the device manager
    void *getDeviceManager(void);
    
    // underlying network is connected (SET)
    void isConnected(bool connected); 
    
    // underlying network is connected (GET)
    bool isConnected();
    
    // Registered with mDC/mDS
    bool isRegistered();

private:
    Logger            *m_logger;
    Options           *m_options;
    bool			   m_canActAsRouterNode;
    bool               m_connected;
    bool			   m_registered;
    
    // mbed-client support
    M2MInterface      *m_interface;
    M2MSecurity       *m_server_instance;
    M2MObjectList      m_object_list;
    M2MDevice         *m_device_object;
    
    // mbed-client methods
    void create_interface();
    M2MSecurity *create_server_instance();
    
    // Device Manager
    void			  *m_device_manager;
	
	// DynamicResource Lookup
	DynamicResource *lookupDynamicResource(M2MBase *base);
};

} // namespace Connector

#endif // __CONNECTOR_ENDPOINT_H__
