/**
 * @file    DeviceManager.cpp
 * @brief   mbed CoAP Endpoint Device Management class
 * @author  Doug Anson
 * @version 1.0
 * @see
 *
 * Copyright (c) 2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
 // configuration
 #include "mbed-connector-interface/mbedConnectorInterface.h"

 // BaseClass 
 #include "mbed-connector-interface/DeviceManager.h"
 
 // Endpoint Class
 #include "mbed-connector-interface/ConnectorEndpoint.h"
 
 // Options Builder
 #include "mbed-connector-interface/OptionsBuilder.h"
 
 // Device Management Responder
 #include "mbed-connector-interface/DeviceManagementResponder.h"
 
// Constructor
DeviceManager::DeviceManager(const Logger *logger,const void *dm_responder,const char *mfg,const char *dev_type,const char *model,const char *serial,const char *fw_vers,const char *hw_vers,const char *sw_vers) {
    // record the data management responder if we have one
    this->m_dm_responder = (void *)dm_responder;
    
    // establish the default base LWM2M device info resource values
    this->m_dev[0] = new DeviceResource(logger,M2MDevice::Manufacturer,mfg);            // Manufacturer
    this->m_dev[1] = new DeviceResource(logger,M2MDevice::DeviceType,dev_type);         // Device Type
    this->m_dev[2] = new DeviceResource(logger,M2MDevice::ModelNumber,model);           // Device Model
    this->m_dev[3] = new DeviceResource(logger,M2MDevice::SerialNumber,serial);         // Device Serial
    this->m_dev[4] = new DeviceResource(logger,M2MDevice::FirmwareVersion,fw_vers);     // Firmware Version
    this->m_dev[5] = new DeviceResource(logger,M2MDevice::HardwareVersion,hw_vers);     // Hardware Version
    this->m_dev[6] = new DeviceResource(logger,M2MDevice::SoftwareVersion,sw_vers);     // Software Version
    
    // record the dev type and logger for later...
    this->m_dev_type = (char *)dev_type;
    this->m_logger = (Logger *)logger;
    
    // Device DeRegistration Resource
    this->m_deregister_resource = new DeviceDeRegisterResource(logger,LWM2M_DEVICE_OBJ_ID,LWM2M_DEV_DEREGISTER_ID,dm_responder);
    
    // Device Reboot Resource
    this->m_reboot_resource = new DeviceRebootResource(logger,LWM2M_DEVICE_OBJ_ID,this->createResName(M2MDevice::Reboot).c_str(),dm_responder);
    
    // Device Reset Resource
    this->m_reset_resource = new DeviceResetResource(logger,LWM2M_DEVICE_OBJ_ID,this->createResName(M2MDevice::FactoryReset).c_str(),dm_responder);
    
    // Device Firmware Composite Resource
    this->m_firmware_composite_resource = new DeviceFirmwareCompositeResource(logger,LWM2M_FIRMWARE_OBJ_ID,dm_responder);
    this->m_firmware_composite_resource->buildResources();
}

// create a string form of our device resource value
string DeviceManager::createResName(M2MDevice::DeviceResource res) {
    char buf[10];
    memset(buf,0,10);
    sprintf(buf,"%d",res); 
    return string(buf);
}

// Copy constructor
DeviceManager::DeviceManager(const DeviceManager &manager) {
    for(int i=0;i<NUM_DEVICE_RESOURCES;++i) {
        this->m_dev[i] = new DeviceResource(*manager.m_dev[i]);
    }
    this->m_dev_type = manager.m_dev_type;
    this->m_logger = manager.m_logger;
    this->m_endpoint = manager.m_endpoint;
    
    // copy the additional resources
}

// Destructor
DeviceManager::~DeviceManager() {
    for(int i=0;i<NUM_DEVICE_RESOURCES;++i) {
        delete this->m_dev[i];
    }
}

// Install the device manager into the Connector Endpoint
void DeviceManager::install(const void *endpoint,const void *config) {
    // record the configuration
    this->m_config = (void *)config;
    
    // record the endpoint
    this->m_endpoint = (void *)endpoint;
    
    // our Endpoint 
    Connector::Endpoint *ep = (Connector::Endpoint *)this->m_endpoint;
    
    // our Endpoint configuration
    Connector::OptionsBuilder *cfg = (Connector::OptionsBuilder *)this->m_config;
    
    // add the base device resources
    for(int i=0;i<NUM_DEVICE_RESOURCES;++i) {
        // DEBUG
        this->m_logger->log("Adding Base Device Resource: %s = %s...",this->m_dev[i]->getFullName().c_str(),this->m_dev[i]->getValue().c_str());
        
        // add the resource
        cfg->addResource(this->m_dev[i]);
    }
    
    // set our device type
    cfg->setEndpointType(this->m_dev_type);
    
    // establish connection to our responder
    if (this->m_dm_responder != NULL) {
        ((DeviceManagementResponder *)this->m_dm_responder)->setEndpoint((const void *)ep);
    }
    
    // add the device action resources
    cfg->addResource(this->m_deregister_resource);           // de-registration action
    cfg->addResource(this->m_reboot_resource);               // reboot action
    cfg->addResource(this->m_reset_resource);                // reset action
    this->m_firmware_composite_resource->addResources(cfg);  // Firmware Composite Resource adds internally...
}

// get our device management responder
void *DeviceManager::getResponder() { return this->m_dm_responder; }