/**
 * @file    DeviceManager.h
 * @brief   mbed CoAP Endpoint Device Management class
 * @author  Doug Anson
 * @version 1.0
 * @see
 *
 * Copyright (c) 2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __DEVICE_MANAGER_H__
#define __DEVICE_MANAGER_H__

// mbed-client support
#include "mbed-client/m2mconstants.h"
 
// Support for Device Resources
#include "mbed-connector-interface/DeviceResource.h"

// Action Resource: Device DeRegistration
#include "mbed-connector-interface/DeviceDeRegisterResource.h"

// Action Resource: Device Reboot
#include "mbed-connector-interface/DeviceRebootResource.h"

// Action Resource: Device Reset
#include "mbed-connector-interface/DeviceResetResource.h"

// Action Resource: Device Firmware
#include "mbed-connector-interface/DeviceFirmwareCompositeResource.h"

// LWM2M Device Info: Mfg, DevType, Model, Serial, Firmware version, Hardware version, Software version
#define NUM_DEVICE_RESOURCES                7

// LWM2M Device Object ID
#define LWM2M_DEVICE_OBJ_ID                 "3"

// LWM2M Firmware Object ID
#define LWM2M_FIRMWARE_OBJ_ID               "5"

// TEMP: Device DeRegistration Resource ID
#define LWM2M_DEV_DEREGISTER_ID             "86"

/** DeviceManager is the endpoint management class
 */
class DeviceManager
{    
    public:
        /**
        Default constructor
        @param logger input logger instance
        @param dm_responder input data management responder/processor
        @param mfg input endpoint manufacturer
        @param dev_type input the endpoint type
        @param model input the model of the endpoint
        @param serial input the serial of the endpoint
        @param fw_vers input the current firmware version
        @param hw_vers input the current hardware version
        @param sw_vers input the current software version
        */
        DeviceManager(const Logger *logger,const void *dm_responder=NULL,const char *mfg="ARM",const char *dev_type="mbed",const char *model="ARM" ,const char *serial="000000",const char *fw_ver="0.0.0",const char *hw_ver="0.0.0",const char *sw_ver="0.0.0");
        
        /**
        Copy constructor
        @param resource input the DeviceManager that is to be deep copied
        */
        DeviceManager(const DeviceManager &manager);
    
        /**
        Destructor
        */
        virtual ~DeviceManager();
        
        /**
        install the device manager into the endpoint
        @param endpoint input the endpoint instance
        @param config input the endpoint configuration instance
        */
        void install(const void *endpoint,const void *config);
        
        /**
        get the DeviceManagementResponder
        @return the DeviceManagementResponder or NULL
        */
        void *getResponder();
        
    private:
        Logger                              *m_logger;
        void                                *m_endpoint;
        void                                *m_config;
        char                                *m_dev_type;
        void                                *m_dm_responder;
        
        // Static LWM2M Device Info Resources
        DeviceResource                      *m_dev[NUM_DEVICE_RESOURCES];
        
        // DM Action-able Resources
        DeviceDeRegisterResource            *m_deregister_resource; 
        DeviceRebootResource                *m_reboot_resource; 
        DeviceResetResource                 *m_reset_resource; 
        DeviceFirmwareCompositeResource     *m_firmware_composite_resource; 
        
        // convenience method to convert enum (int) type to string 
        string                               createResName(M2MDevice::DeviceResource res);
};

#endif // __DEVICE_MANAGER_H__