/**
 * @file    DeviceManagementResponder.h
 * @brief   mbed CoAP Endpoint Device Management Responder class
 * @author  Doug Anson
 * @version 1.0
 * @see
 *
 * Copyright (c) 2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __DEVICE_MANAGEMENT_RESPONDER_H__
#define __DEVICE_MANAGEMENT_RESPONDER_H__

// Logger
#include "mbed-connector-interface/Logger.h"

// Authenticator Support
#include "mbed-connector-interface/Authenticator.h"

// invocation handler signature
typedef bool (*responder_fn)(const void *ep,const void *logger,const void *data); 

class DeviceManagementResponder {
    public:
        /**
        Default constructor
        @param logger input logger instance
        @param authenticator input authentication instance
        */
        DeviceManagementResponder(const Logger *logger,const Authenticator *authenticator);
        
        /**
        Copy constructor
        @param resource input the DeviceManagementResponder that is to be deep copied
        */
        DeviceManagementResponder(const DeviceManagementResponder &manager);
    
        /**
        Destructor
        */
        virtual ~DeviceManagementResponder();
        
        /**
        Set the Endpoint instance
        @param ep input the endpoint instance pointer
        */
        void setEndpoint(const void *ep);
        
        /** 
        Set our Reboot Responder handler function
        @param reboot_responder_fn input the device reboot responder function pointer
        */
        void setRebootResponderHandler(responder_fn reboot_responder_fn);
        
        /** 
        Set our Reset Responder handler function
        @param reset_responder_fn input the device reset responder function pointer
        */
        void setResetResponderHandler(responder_fn reset_responder_fn);
        
        /** 
        Set our FOTA invocation handler function
        @param fota_invocation_fn input the FOTA invocation handler function pointer
        */
        void setFOTAInvocationHandler(responder_fn fota_invocation_fn);
        
        /**
        Set our FOTA manifest
        @param fota_manifest input the input FOTA manifest
        */
        virtual void setFOTAManifest(const char *fota_manifest);
        
        /**
        Get our FOTA manifest
        @return the FOTA manifest
        */
        virtual char *getFOTAManifest();
        
        /**
        ACTION: Deregister device
        @param challenge input the input authentication challenge
        */
        virtual void deregisterDevice(const void *challenge);
        
        /**
        ACTION: Reboot device
        @param challenge input the input authentication challenge
        */
        virtual void rebootDevice(const void *challenge);
        
        /**
        ACTION: Reset device 
        @param challenge input the input authentication challenge
        */
        virtual void resetDevice(const void *challenge);
        
        /**
        ACTION: Invoke FOTA (default: empty action)
        @param challenge input the input authentication challenge
        */
        virtual void invokeFOTA(const void *challenge);
        
        /**
        Set the FirmwareComposite Resource
        @param firmware_composite_resource input the Firmware composite resource instance
        */
        void setFirmwareCompositeResource(const void *firmware_composite_resource);
        
        /**
        Get the FirmwareComposite Resource
        @return the Firmware composite resource or NULL
        */
        void *getFirmwareCompositeResource();
    
    private:
        Logger                      *m_logger;
        Authenticator               *m_authenticator;
        void                        *m_endpoint;
        char                        *m_fota_manifest;
        
        responder_fn                 m_reboot_responder_fn;
        responder_fn                 m_reset_responder_fn;
        responder_fn                 m_fota_invocation_fn;
        
        bool                         authenticate(const void *challenge);
        
        void                        *m_firmware_composite_resource;
};

#endif // __DEVICE_MANAGEMENT_RESPONDER_H__