/* Copyright C2013 Doug Anson, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 // class definition
 #include "NSPResourceFactory.h"
 
 // NSP resource support
 #include "NSPResource.h"
      
 // default constructor
 NSPResourceFactory::NSPResourceFactory(Logger *logger,void *endpoint) : ResourceFactory(logger,endpoint) {
     memset(this->m_endpoint_name,0,PERSONALITY_NAME_LEN+1);
     this->m_resource_ptr = NULL;
 }
 
 // default destructor
 NSPResourceFactory::~NSPResourceFactory() {
 }
 
 // set the endpoint name
 void NSPResourceFactory::setEndpointName(char *ep_name) { 
     memset(this->m_endpoint_name,0,PERSONALITY_NAME_LEN+1);
     strncpy(this->m_endpoint_name,ep_name,this->min(strlen(ep_name),PERSONALITY_NAME_LEN)); 
 }
 
 // get the endpoint name
 char *NSPResourceFactory::getEndpointName() { return this->m_endpoint_name; }
 
 // send any observations we have
 void NSPResourceFactory::sendObservations() {
     for(int i=0;i<this->m_count;++i) {
         NSPResource *resource = (NSPResource *)this->m_list[i];
         if (resource != NULL) {
             resource->sendObservation();
         }
     }
 }
 
 // create an NSP resource
 void NSPResourceFactory::createResource(char *name,char *value) { this->createResource(name,value,(char *)NULL,(char *)NULL); }
 void NSPResourceFactory::createResource(char *name,char *value,char *interface,char *resource) { 
    this->createResource(this->getEndpointName(),name,value,NULL,SN_GRS_GET_ALLOWED,NULL,interface,resource);
 }
 void NSPResourceFactory::createResource(char *ep_name,char *name,char *value,void *cb) { this->createResource(ep_name,name,value,cb,(char *)NULL,(char *)NULL); }
 void NSPResourceFactory::createResource(char *ep_name,char *name,char *value,void *cb,char *interface, char *resource) { 
    this->createResource(ep_name,name,value,cb,SN_GRS_GET_ALLOWED,NULL,interface,resource); 
 }
 void NSPResourceFactory::createResource(char *ep_name,char *name,char *value,void *cb,sn_grs_resource_acl_e acl) {
     this->createResource(ep_name,name,value,cb,acl,(char *)NULL,(char *)NULL); 
 }
 void NSPResourceFactory::createResource(char *ep_name,char *name,char *value,void *cb,sn_grs_resource_acl_e acl,char *interface, char *resource) {
     this->createResource(ep_name,name,value,cb,acl,NULL,interface,resource);
 }
 void NSPResourceFactory::createResource(char *ep_name,char *name,char *value,void *cb,sn_grs_resource_acl_e acl,void *io) {
     this->createResource(ep_name,name,value,cb,acl,io,(char *)NULL,(char *)NULL);
 }
 void NSPResourceFactory::createResource(char *ep_name,char *name,char *value,void *cb,sn_grs_resource_acl_e acl,void *io,char *interface, char *resource) {     
     // initialize the resource
     this->m_list[this->m_count] = new NSPResource(this->logger(),ep_name,name,value,cb,this->m_resource_ptr,acl,interface,resource); 
     
     // call the resource initializer if needed
     if (io != NULL && this->m_list[this->m_count] != NULL) {
         //this->logger()->log("Calling initializer for %s...",this->m_list[this->m_count]->getName());
         resourceInitializer initializer = (resourceInitializer)io;
         (initializer)(this->m_list[this->m_count]);
     }
     
     // increment the counter
     if (this->m_list[this->m_count] != NULL) ++this->m_count;
 }
 
 // initialize resource structures if needed
 void NSPResourceFactory::initResourceStructures() {
     if (this->m_resource_ptr == NULL) {
         this->m_resource_ptr = (void *)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
         if (this->m_resource_ptr != NULL) {
             memset(this->m_resource_ptr,0,sizeof(sn_nsdl_resource_info_s));
             ((sn_nsdl_resource_info_s *)this->m_resource_ptr)->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
             if (((sn_nsdl_resource_info_s *)this->m_resource_ptr)->resource_parameters_ptr != NULL) {
                 memset(((sn_nsdl_resource_info_s *)this->m_resource_ptr)->resource_parameters_ptr,0,sizeof(sn_nsdl_resource_parameters_s));
             }
             else {
                 this->logger()->log("NSP: malloc of NSP resource_parameters_ptr FAILED");
                 this->freeResourceStructures();
             }
         }
         else {
             this->logger()->log("NSP: malloc of NSP resource_ptr FAILED");
             this->freeResourceStructures();
         }
     }
 }
 
 // free resource structures
 void NSPResourceFactory::freeResourceStructures() {     
     if (this->m_resource_ptr != NULL) {
         nsdl_free(((sn_nsdl_resource_info_s *)this->m_resource_ptr)->resource_parameters_ptr);
         nsdl_free((sn_nsdl_resource_info_s *)this->m_resource_ptr); 
     } 
     this->m_resource_ptr = NULL;
 }
 
 // finalize the registrations
 void NSPResourceFactory::finalizeRegistration() {
     sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
     
     // register the endpoint and its resources with NSP
     endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr,(uint8_t*)_endpoint_name,_ep_type,_lifetime_ptr);
     if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
        this->logger()->log("NSP: Registration of resources for %s FAILED",this->getEndpointName());
     else
        this->logger()->log("Installed %d NSP resources",this->m_count);
     
     // clean up
     if (endpoint_ptr != NULL) nsdl_clean_register_endpoint(&endpoint_ptr);
     this->freeResourceStructures();   
 }