/* Copyright C2013 Doug Anson, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
#ifndef _MBED_ENDPOINT_H_
#define _MBED_ENDPOINT_H_

// BaseClass support
#include "BaseClass.h"

// NSP Transport support
#include "NSPTransport.h"

// HTTP Transport support
#include "HTTPTransport.h"

// Personality Support
#include "Personality.h"

// Preferences Support
#include "Preferences.h"

// network selection
#ifdef CELLULAR_NETWORK
    // Cellular
    #include "MBEDUbloxCellRadio.h"
    #include "MBEDUbloxGPS.h"
#else
    // Ethernet
    #include "EthernetInterface.h"
#endif
  
class MBEDEndpoint : public BaseClass {
    private:
#ifdef CELLULAR_NETWORK
        MBEDUbloxCellRadio   *m_cellular_modem;                            // cell modem
        MBEDUbloxGPS         *m_gps;                                       // GPS receiver
#else
        EthernetInterface    *m_ethernet;                                  // ethernet interface
#endif

        Transport         *m_transports[NUM_TRANSPORTS];                    // our transport
        Personality       *m_personalities[NUM_PERSONALITY_INSTANCES];      // our personalities (at least 1 instance)
        char               m_endpoint_name[PERSONALITY_NAME_LEN+1];         // our endpoint name (personalities[0])
        Preferences       *m_preferences;                                   // preference support
        char               m_lcd_status[TEMP_BUFFER_LEN+1];                 // LCD status buffer
        char               m_nsp_address[PREFERENCE_VALUE_LEN+1];           // NSP address buffer
        char               m_nsp_port_str[PREFERENCE_VALUE_LEN+1];          // NSP address port buffer
        char               m_nsp_port_str_def[PREFERENCE_VALUE_LEN+1];      // NSP address port buffer (default)
        int                m_nsp_port;                                      // NSP port
        int                m_instance_id;                                   // Instance ID for this endpoint
        void              *m_status_reporter;                               // Status Reporter
        
    public:
        MBEDEndpoint(Logger *logger,void *transport,void *status_reporter,void *extra);
        virtual ~MBEDEndpoint();
        ResourceFactory *initResourceFactory();
        void run();
        
        char *getEndpointName();
        char *getNSPAddress();
        int getNSPPort();
        Preferences *preferences();
        
        char *getLCDStatus();
        
        int   getInstanceID();
        
        void  sendObservations();
        
        void  extraEventLoopWork();
        
    private:
        void initPreferences();
        void initNSPAddress();
        void initEndpointName();
        bool initializePersonalities();
        bool initializeLights();
        bool initializeTransport(int index,char *key,Transport *transport);
        bool initializeTransports();
        bool closePersonalities();
        bool closeTransport(int index,char *key);
        bool closeTransports();
        
        bool autoRegisterWithIOC();
        
#ifdef CELLULAR_NETWORK
        bool initializeCellularModem(void *modem);
        bool initializeGPSReceiver(void *gps);
        bool closeCellularModem();
        bool closeGPSReceiver();
#else
        bool initializeEthernet(EthernetInterface *ethernet);
        bool closeEthernet();
#endif 
};

#endif // _MBED_ENDPOINT_H_
