/* Copyright C2013 Doug Anson, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 #include "MBEDEndpoint.h"
 
 #include <string.h>
 #include <stdlib.h>
 
 #ifdef CELLULAR_NETWORK
    // MDMSerial 
 #ifdef NSP_CELLULAR_NETWORK
    #include "mbed.h"
    #include "rtos.h"
 #endif
    #include "MDM.h"
    
    // GPSParser
    #include "GPS.h"
 #else
    // Ethernet Interface
    #include "EthernetInterface.h"
    EthernetInterface ethernet;
 #endif
 
 #ifdef NETWORK_MUTEX 
 // Mutex for the network as RTOS doesnt play well with threads...
 Mutex *network_mutex = NULL;
 #endif
 
 // Buffered Serial
 #include "BufferedSerial.h"

 // Serial Console Support
 BufferedSerial pc(USBTX, USBRX);
 
 #ifdef _ENDPOINT_UBLOX_PLATFORM    
     // LCD Support
     #include "C12832.h"
     C12832 lcd(D11, D13, D12, D7, D10);
#endif

#ifdef _ENDPOINT_NXP_PLATFORM     
     // LCD Support
     #include "C12832_lcd.h"
     C12832_LCD lcd; 
#endif

#ifdef _ENDPOINT_FREEDOM_PLATFORM        
     // LCD Support
     #include "C12832_lcd.h"
     C12832_LCD lcd;  
#endif
 
 // Instances for the Endpoint
 Logger *m_logger = NULL;
 MBEDEndpoint *endpoint = NULL;
 
 // HTTP support
 HTTPClient _http;
 
 #if SF_STATUS_REPORTING
    // include Saleforce status reporting
    #include "StatusReporter.h"
 #endif
 void *sf_status_reporter = NULL;
 
 #ifdef MAC_ADDRESS
 char mac[6] = {MAC_ADDRESS};
 extern "C" void mbed_mac_address(char *s) { for(int i=0;i<6;++i) s[i] = mac[i]; }
 char fmt_mac[RESOURCE_VALUE_LEN+1];
 #endif
 
 // Logger (logger)
 Logger *logger() { return m_logger; }    
 
 // translate a closedown code to a string
 char str_code[16];
 char *strCode(int code) {
     memset(str_code,0,16);
     if (code == 1) strcpy(str_code,"(USER)");
     if (code == 2) strcpy(str_code,"(ERROR)");
     return str_code;
 } 

 // close down the application and exit
 void closedown(int code) {
     if (endpoint != NULL) {
#ifdef EH_USE_MUTEXES  
         m_logger->releaseMutexes();
#endif
         m_logger->log("Closing down Endpoint...");
         delete endpoint;
     }
     if (m_logger != NULL) delete m_logger;
#ifndef _ENDPOINT_UBLOX_PLATFORM
     pc.printf("Exiting...\r\n");
#endif
     lcd.cls();
     lcd.locate(0,0);
     lcd.printf("Endpoint Shutdown %s",strCode(code));
     exit(1);
 }
 
 // strdup()
 char *strdup(char *str) {
     if (str != NULL) {
         char *cp = (char *)malloc(strlen(str)+1);
         memset(cp,0,strlen(str)+1);
         memcpy(cp,str,strlen(str));
         return cp;
     }
     return NULL;
 }
 
 // HARD RESET - ugly but necessary - we seem to develop memory holes.... :(
 extern "C" void HardFault_Handler() { NVIC_SystemReset(); }
 
 // main entry point
 int main() {      
#ifdef NETWORK_MUTEX
    network_mutex = new Mutex();
    if (network_mutex != NULL) network_mutex->unlock();
#endif
#ifdef MAC_ADDRESS
    mbed_mac_address(mac);
    memset(fmt_mac,0,RESOURCE_VALUE_LEN+1);
    sprintf(fmt_mac,"%02x:%02x:%02x:%02x:%02x:%02x",mac[0],mac[1],mac[2],mac[3],mac[4],mac[5]);
#endif
    m_logger = new Logger(&pc,&lcd);
    
#if SF_STATUS_REPORTING
    StatusReporter status_reporter(&_http,m_logger);
    sf_status_reporter = (void *)&status_reporter;
#endif

#ifdef CELLULAR_NETWORK
    #ifdef NSP_CELLULAR_NETWORK
        MDMRtos<MDMSerial> modem;
    #else
        MDMSerial modem;
    #endif
    endpoint = new MBEDEndpoint(m_logger,&modem,sf_status_reporter,NULL);
#else
    endpoint = new MBEDEndpoint(m_logger,&ethernet,sf_status_reporter,NULL);
#endif
    if (endpoint != NULL) endpoint->run();
 }
