/* Copyright C2013 Doug Anson, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 #include "mbed.h"
 #include "Preferences.h"
 
 #ifdef _ENDPOINT_UBLOX_PLATFORM
    #define NO_LOCAL_FILESYSTEM     true
 #endif
 
 #ifndef NO_LOCAL_FILESYSTEM
 LocalFileSystem local("local");
 #endif
 
 // default constructor
 Preferences::Preferences(Logger *logger) : BaseClass(logger,NULL) {
     this->m_num_preferences = 0;
#ifndef NO_LOCAL_FILESYSTEM  
     this->initialize();
#endif
 }
 
 // destructor
 Preferences::~Preferences() {
 }
 
 // initialize the preferences from the preferences file
 bool Preferences::initialize() {
     bool success = true;
                       
     char name[PREFERENCE_NAME_LEN+1];
     char value[PREFERENCE_VALUE_LEN+1];
     char buffer[PREFERENCE_NAME_LEN + PREFERENCE_VALUE_LEN + 10];

     // initialize the memory buffers and count
     for(int i=0;i<MAX_NUM_PREFERENCES;++i) {
         memset(this->m_preferences[i].name,0,PREFERENCE_NAME_LEN+1);
         memset(this->m_preferences[i].value,0,PREFERENCE_VALUE_LEN+1);
     }
   
     // initialize temp buffers
     memset(name,0,PREFERENCE_NAME_LEN+1);
     memset(value,0,PREFERENCE_VALUE_LEN+1);
     memset(buffer,0,PREFERENCE_NAME_LEN + PREFERENCE_VALUE_LEN + 10);
 
     // read and open the config file
     FILE *fp = fopen(PREFERENCES_FILE,"r");
     if (fp != NULL) {
         // read in the first line
         memset(buffer,0,PREFERENCE_NAME_LEN + PREFERENCE_VALUE_LEN + 10);
         int n = fscanf(fp,"%s",buffer);
         
         // loop and read each line
         while(n != EOF) {
             // replace the equals sign with a space
             for(int i=0;i<strlen(buffer);++i) if (buffer[i] == '=') buffer[i] = ' ';
             
             // parse
             memset(name,0,PREFERENCE_NAME_LEN+1);
             memset(value,0,PREFERENCE_VALUE_LEN+1);
             sscanf(buffer,"%s%s",name,value);
             strcpy(this->m_preferences[this->m_num_preferences].name,name);
             strcpy(this->m_preferences[this->m_num_preferences].value,value);
             
             // DEBUG
             this->logger()->log("Installed Preference: %s=%s",this->m_preferences[this->m_num_preferences].name,this->m_preferences[this->m_num_preferences].value);
             
             // increment the tally
             ++this->m_num_preferences;
             
             // reset the buffer and read in another line
             memset(buffer,0,PREFERENCE_NAME_LEN + PREFERENCE_VALUE_LEN + 10);
             n = fscanf(fp,"%s",buffer);
         }
         
         // close
         fclose(fp);
         
         // summary
         this->logger()->log("Installed %d preferences",this->m_num_preferences);
     }
     else {
         // unable to open the preferences file - not loaded
         this->logger()->log("Unable to open preferences file %s... Preferences not loaded.",PREFERENCES_FILE);
         success = false;
     }
     
     // return our status
     return success;
 }
 
 // HACK: fix up coords because IOC's POINT() macro does not like commas
 void Preferences::fixCoordsForIOC() {
     int index = this->indexOfPreference("coords");
     if (index >= 0) {
         // remove the comma...
         for(int i=0;i<strlen(this->m_preferences[index].value);++i) if (this->m_preferences[index].value[i] == ',') this->m_preferences[index].value[i] = ' ';
     }
 }
 
 // integer preference with defaults
 int Preferences::getIntPreference(char *name,int def_value) {
     int int_value = def_value;
     char buffer[PREFERENCE_VALUE_LEN+1];
     memset(buffer,0,PREFERENCE_VALUE_LEN+1);
     char *value = this->getPreference(name,buffer,PREFERENCE_VALUE_LEN,NULL);
     if (value != NULL && strlen(value) > 0) sscanf(buffer,"%d",&int_value);
     return int_value;
 }
  
 // boolean preference with defaults
 bool Preferences::getBooleanPreference(char *name,bool def_value) {
     bool bool_value = def_value;
     char buffer[PREFERENCE_VALUE_LEN+1];
     memset(buffer,0,PREFERENCE_VALUE_LEN+1);
     char *value = this->getPreference(name,buffer,PREFERENCE_VALUE_LEN,NULL);
     if (value != NULL && strcmp(value,"true") == 0) bool_value = true;
     if (value != NULL && strcmp(value,"false") == 0) bool_value = false;
     return bool_value;
 }
 
 // string preference with defaults
 char *Preferences::getPreference(char *name,char *buffer,int buffer_length,char *def_value) {
     char *value = NULL;
     
     if (buffer != NULL && buffer_length > 0) {
         // clean the buffer
         memset(buffer,0,buffer_length);
         
         // DEBUG
         //this->logger()->log("GET: %s  DEFAULT: %s  BUFFER_LEN=%d",name,def_value,buffer_length);
         
         // get our index
         int index = this->indexOfPreference(name);
         if (index >= 0) {
             // fill with our value
             strncpy(buffer,this->m_preferences[index].value,this->min(buffer_length,strlen(this->m_preferences[index].value)));
             value = buffer;
             //this->logger()->log("Preference GET: %s=%s",name,value);
         }
         else {
             // fill with the default
             if (def_value != NULL) {
                 strncpy(buffer,def_value,this->min(buffer_length,strlen(def_value)));
                 value = buffer;
                 //this->logger()->log("Using Defaults for %s=%s",name,value);
             }
             else {
                 // no defaults
                 //this->logger()->log("No default for GET: %s",name);
                 value = buffer;
             }
         }
     }
     
     // return the value
     return value;
 }
 
 // get the index of the named preference
 int Preferences::indexOfPreference(char *name) {
     bool found = false;
     int index = -1;
     
     // parameter check
     if (name != NULL && strlen(name) > 0 && this->m_num_preferences > 0) {
         // loop until we find the name.. then stop and record its index
         for(int i=0;i<this->m_num_preferences && !found;++i) {
             if (strcmp(name,this->m_preferences[i].name) == 0) {
                found = true;
                index = i;
             }
         }
     }
     
     // return the index value
     return index;
 }
  
 // get our preference count
 int Preferences::numPreferences() { return this->m_num_preferences; }