/* Copyright C2013 Doug Anson, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 #include "Light.h"
 #include "MBEDEndpoint.h"
 
 // include the relevant action support
 #include "LightDimmerAction.h"
 #include "LightSwitchAction.h"
 
 void *_instance = NULL;
 
 // Blinking Looper
 static void blinking_action(void const *args) {
     if (_instance != NULL) {
        Light *self = (Light *)_instance;
        self->updateDirection();
        while(true) self->blinkLight();
    }
 }
 
 // default constructor
 Light::Light(Logger *logger,Transport *transports[NUM_TRANSPORTS],int instance,void *endpoint) : Personality(logger,transports,instance,endpoint,LIGHT_TYPE_STRING) {
     this->m_forward = true;
      
     // setup the blinking thread
 #ifdef ENABLE_THREADS
     this->m_blinking_thread = NULL;
 #endif
     this->m_is_blinking = false;
     
     // Setup Philips Light if enabled
     if (PL_ENABLE) this->m_pl = new PhilipsLight(PL_LIGHT_ID,this->m_transports[HTTP_TRANSPORT],this->logger());
     else this->m_pl = NULL;
 
 #ifdef COPCAR_PERSONALITY
     // Setup External CopCar Flasher if enabled
     if (EXT_LED_ENABLE) this->m_ext_led = new CopCarLEDFlasher(new PwmOut(EXT_LED_PIN),new PwmOut(EXT_LED_PIN_COPCAR),this->logger());
     else this->m_ext_led = NULL;
 #endif
 
 #ifdef LIGHT_PERSONALITY    
     // Setup External LED Light if enabled
     if (EXT_LED_ENABLE) this->m_ext_led = new ExternalLEDLight(new PwmOut(EXT_LED_PIN),this->logger());
     else this->m_ext_led = NULL;
 #endif
 
 #ifdef APM_LIGHT_PERSONALITY    
     // Setup APM Light if enabled
     if (APM_LIGHT_ENABLE) this->m_apm_light = new APMDemoLight(this->logger());
     else this->m_apm_light = NULL;
 #endif
          
     // DEBUG
     if (PL_ENABLE) this->logger()->log("Light name: %s (Philips Light %d)",this->getName(),PL_LIGHT_ID);   
     else this->logger()->log("Light name: %s", this->getName()); 
          
     // we are activated
     _instance = (void *)this;
 }
 
 // destructor
 Light::~Light() {   
    if (this->m_dimmer_action != NULL) delete this->m_dimmer_action;
    if (this->m_switch_action != NULL) delete this->m_switch_action;
    if (this->m_pl != NULL) delete this->m_pl;
    if (this->m_ext_led != NULL) delete this->m_ext_led;
 #ifdef APM_LIGHT_PERSONALITY
    if (this->m_apm_light != NULL) delete this->m_apm_light;
 #endif
    this->stopBlinkingThread();
 }
 
 // initialize the light
 void Light::initLight() {
     sscanf(LIGHT_DEFAULT_STATE,"%d",&this->m_current_state);
     this->m_last_state = this->m_current_state;
 }
 
 // get the Philips light
 PhilipsLight *Light::pl() { return this->m_pl; }
 
 // get the External LED light
 ExternalLEDLight *Light::extled() { return this->m_ext_led; }
 
 #ifdef APM_LIGHT_PERSONALITY
 // get the APM light
 APMDemoLight *Light::apmlight() { return this->m_apm_light; }
 #endif
 
 // set the dimmer action
 void Light::setDimmerAction(void *dimmer_action) { this->m_dimmer_action = dimmer_action; }
 
 // set the switch actino
 void Light::setSwitchAction(void *switch_action) {this->m_switch_action = switch_action; }
 
 // get the dimmer action
 void *Light::getDimmerAction() { return this->m_dimmer_action; }
 
 // get the switch action
 void *Light::getSwitchAction() { return this->m_switch_action; }
    
 // turn ON 
 void Light::on() { 
    this->m_current_state = 1; 
    this->manageBlinkingThread();  
    if (PL_ENABLE && this->pl() != NULL) this->pl()->on(); 
    if (EXT_LED_ENABLE && this->extled() != NULL) this->extled()->on();
#ifdef APM_LIGHT_PERSONALITY
    if (APM_LIGHT_ENABLE && this->apmlight() != NULL) this->apmlight()->on();
#endif
 }
 
 // turn OFF 
 void Light::off() {  
    this->m_current_state = 0; 
    this->manageBlinkingThread(); 
    if (PL_ENABLE && this->pl() != NULL) this->pl()->off(); 
    if (EXT_LED_ENABLE && this->extled() != NULL) this->extled()->off();
#ifdef APM_LIGHT_PERSONALITY
    if (APM_LIGHT_ENABLE && this->apmlight() != NULL) this->apmlight()->off();
#endif
 }
 
 // initiate blinking
 void Light::blink() {
     this->m_last_state = this->m_current_state;
#ifdef APM_COPCAR_ENABLE
     // just enable the external pin to go high
     if (EXT_LED_ENABLE && this->extled() != NULL) { ((CopCarLEDFlasher *)this->extled())->on(); }
#else
     this->startBlinkingThread();
#endif
  }
 
 // manage the blinking thread
 void Light::manageBlinkingThread() {
     if (this->m_is_blinking == false) this->stopBlinkingThread();
     this->m_is_blinking = false;
 }
 
 // stop blinking
 void Light::stopBlinking() {
     this->m_is_blinking = false;
     if (this->m_last_state == 1) this->on();
     if (this->m_last_state == 0) this->off();
     this->m_current_state = this->m_last_state;
#ifdef APM_COPCAR_ENABLE
     if (EXT_LED_ENABLE && this->extled() != NULL) { ((CopCarLEDFlasher *)this->extled())->off(); }
#endif
 }
 
 // start blinking thread
 void Light::startBlinkingThread() {
 #ifdef ENABLE_THREADS
     if (this->m_blinking_thread == NULL)
        this->m_blinking_thread = new Thread(blinking_action);
 #endif
 }
 
 // stop blinking thread
 void Light::stopBlinkingThread() { 
 #ifdef ENABLE_THREADS
    if (this->m_blinking_thread != NULL) {
        this->m_blinking_thread->terminate(); 
        delete this->m_blinking_thread;
    }
    this->m_blinking_thread = NULL;
 #endif
 }
 
 // update our blinking sequencing to properly mesh with the lights current state
 void Light::updateDirection() {
    this->m_forward = true;
    if (this->m_current_state == 0) this->m_forward = false;
 }
 
 // Blink 
 void Light::blinkLight() { 
    this->m_is_blinking = true;
#ifdef COPCAR_PERSONALITY
    if (EXT_LED_ENABLE && this->extled() != NULL) { ((CopCarLEDFlasher *)this->extled())->on(); }
    Thread::wait(125);
    if (EXT_LED_ENABLE && this->extled() != NULL) { ((CopCarLEDFlasher *)this->extled())->off(); }
    this->m_current_state = 0; 
    this->manageBlinkingThread();
#endif
#ifdef LIGHT_PERSONALITY
    if (this->m_forward) this->on(); else this->off();
    if (PL_ENABLE && this->pl() != NULL) { if (this->m_forward) this->pl()->on(); else this->pl()->off(); }
    if (EXT_LED_ENABLE && this->extled() != NULL) { if (this->m_forward) this->extled()->on(); else this->extled()->off(); }
#endif
#ifdef APM_LIGHT_PERSONALITY
    if (APM_LIGHT_ENABLE && this->apmlight() != NULL) { if (this->m_forward) this->apmlight()->on(); else this->apmlight()->off(); }
#endif

    Thread::wait(LIGHT_BLINK_WAIT_MS);
    this->m_is_blinking = true;
#ifdef COPCAR_PERSONALITY
    if (EXT_LED_ENABLE && this->extled() != NULL) { ((CopCarLEDFlasher *)this->extled())->on(); }
    Thread::wait(125);
    if (EXT_LED_ENABLE && this->extled() != NULL) { ((CopCarLEDFlasher *)this->extled())->off(); }
    this->m_current_state = 0; 
    this->manageBlinkingThread();
#endif
#ifdef LIGHT_PERSONALITY
    if (this->m_forward) this->off(); else this->on();
    if (PL_ENABLE && this->pl() != NULL) { if (this->m_forward) this->pl()->off(); else this->pl()->on(); }
    if (EXT_LED_ENABLE && this->extled() != NULL) { if (this->m_forward) this->extled()->off(); else this->extled()->on(); }
#endif
#ifdef APM_LIGHT_PERSONALITY
    if (APM_LIGHT_ENABLE && this->apmlight() != NULL) { if (this->m_forward) this->apmlight()->off(); else this->apmlight()->on(); }
#endif
    Thread::wait(LIGHT_BLINK_WAIT_MS);
  }
  
 // dim 
 void Light::dim(int value) { 
    if (PL_ENABLE && this->pl() != NULL) this->pl()->dim(value); 
    if (EXT_LED_ENABLE && this->extled() != NULL) this->extled()->dim(value);
#ifdef APM_LIGHT_PERSONALITY
    if (APM_LIGHT_ENABLE && this->apmlight() != NULL) this->apmlight()->dim(value);
#endif
 }
 
 
 