/* Copyright C2013 Doug Anson, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 #include "APMDemoLight.h"
 
 // Configuration settings for the APM Demo Light...
 #ifdef APM_LIGHT_PERSONALITY
  
 // pin for ZeroCross tail input - NOTE: An external 1K pullup required
 InterruptIn apm_light_zerocross(p27);
 
 // pin for PowerSSRtail output
 DigitalOut apm_light_SSR(p21);
 
 //use timer interrupts to control dimming
 Timeout apm_light_SSRtriggerOn;
  
 // AC power line frequency
 const float apm_light_powerlinefrequency = 50.000;
 
 // Our instance
 APMDemoLight *_apm_light_instance = NULL;
 
 // this interrupt routine is activated after a time delay set by dim value
 extern "C" void apm_light_triggerOn()
 {
    apm_light_SSR = 1;
 }
 
 // this interrupt routine is activated by every AC line zero crossing
 // it is needed to synchronize the SCR turnon time delay to the AC line
 extern "C" void apm_light_dimmer()
 {
    // turn off SSR at zero crossing
    apm_light_SSR = 0;
    
    // compute time delay using dim value and set timer interrupt
    // triggers SSR after a small post zero crossing time delay
    apm_light_SSRtriggerOn.attach(&apm_light_triggerOn,(1.001-_apm_light_instance->getDimValue())/(2*apm_light_powerlinefrequency));
 }
   
 // default constructor
 APMDemoLight::APMDemoLight(Logger *logger) : BaseClass(logger,NULL) {
     this->m_dimming = false;
     sscanf(LIGHT_DEFAULT_STATE,"%d",&this->m_state);
     sscanf(LIGHT_DIM_STATE,"%f",&this->m_dim);
     
     // establish our instance
     _apm_light_instance = this;
     
     // ZeroCross configuration
     apm_light_zerocross.mode(PullNone);
     Thread::wait(LIGHT_BLINK_WAIT_MS);
     apm_light_zerocross.rise(&apm_light_dimmer);
 }
 
 // destructor
 APMDemoLight::~APMDemoLight() {
 }
 
 // turn the light on
 void APMDemoLight::on() { this->m_state = 1; this->m_dimming = false; this->update(); }
 
 // turn the light off
 void APMDemoLight::off() { this->m_state = 0; this->m_dimming = false; this->update(); }
  
 // dim the light (dim is between 0 and 100)
 void APMDemoLight::dim(int dim) { this->m_dim = (float)(dim/100.0); this->m_dimming = true; this->update(); }
 
 // get the dimming value
 float APMDemoLight::getDimValue() { return this->m_dim; }
  
 // set the light state
 void APMDemoLight::update() { 
    // handled automatically
    ;
 }
 
 #endif