/**
 * @file    LocationResource.h
 * @brief   mbed CoAP Endpoint Location resource supporting CoAP GET
 * @author  Doug Anson
 * @version 1.0
 * @see
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __LOCATION_RESOURCE_H__
#define __LOCATION_RESOURCE_H__

// Base class
#include "DynamicResource.h"

// main.cpp can enable/disable the experimental location source 
#if ENABLE_BLE_LOCATION
    // our BLE Location source
    #include "BLELocation.h"
    extern RawSerial pc;               // main.cpp
    BLELocation _ble_location(&pc);    // BLE Location from the UART Proxy application
#endif

// We have a static location by default - Moscone West - 37.783879,-122.4012538, altitude 30m msl
#define DEF_LATITUDE          37.783879
#define DEF_LONGITUDE        -122.4012538
#define DEF_ALTITUDE          30.0
#define DEF_SPEED             0.0

// Maximum Location JSON Length : {"latitude":XXX.YYYYYY, "longitude":XXX.YYYYYY, "msl":XXXXXX, "speed":XXXXXX}
#define LOCATION_JSON_LENGTH  256
char __location_json[LOCATION_JSON_LENGTH+1];

/** LocationResource class
 */
class LocationResource : public DynamicResource
{
public:
    /**
    Default constructor
    @param logger input logger instance for this resource
    @param name input the Location resource name
    @param observable input the resource is Observable (default: FALSE)
    */
    LocationResource(const Logger *logger,const char *name,const bool observable = false) : DynamicResource(logger,name,"Location",SN_GRS_GET_ALLOWED,observable) {
        memset(__location_json,0,LOCATION_JSON_LENGTH+1);
#if ENABLE_BLE_LOCATION 
        _ble_location.setDefault(DEF_LATITUDE,DEF_LONGITUDE,DEF_ALTITUDE,DEF_SPEED);
#endif
    }

    /**
    Get the value of the Location sensor
    @returns string containing the location value
    */
    virtual string get() {
        //_ble_location.updateLocation();
        memset(__location_json,0,LOCATION_JSON_LENGTH);
#if ENABLE_BLE_LOCATION
        sprintf(__location_json,"{\"latitude\":%s,\"longitude\":%s,\"msl\":%s,\"speed\":%s,\"src\":\"proxy\"}",
                    _ble_location.getLatitude(),
                    _ble_location.getLongitude(),
                    _ble_location.getMSLAltitude(),    // in meters
                    _ble_location.getSpeed());         // in meters/second
#else
        sprintf(__location_json,"{\"latitude\":%.6f,\"longitude\":%.6f,\"msl\":%.1f,\"speed\":%.1f,\"src\":\"static\"}",
                    DEF_LATITUDE,
                    DEF_LONGITUDE,
                    DEF_ALTITUDE,       // in meters
                    DEF_SPEED);         // in meters/second
#endif
        return string(__location_json);
    }
};

#endif // __LOCATION_RESOURCE_H__