/* Thermostat-LEDUtils.h */
/* Copyright (C) 2013 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
#ifndef THERMOSTAT_LEDUTILS_H_
#define THERMOSTAT_LEDUTILS_H_

// LEDs to toggle with control messages
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED1);
DigitalOut led4(LED2);

// Joystick button (to create an error condition)
DigitalIn joystick_pressed(A2);
DigitalIn joystick(A3);        

// Pot 1 for dialing up and down the temperature
AnalogIn Pot1(A0);

// Pot 2 for dialing up and down the battery level
AnalogIn Pot2(A1);

// the led's are connected to vcc, so a PwmOut of 100% will shut off the led and 0% will let it shine
PwmOut r (D5);
PwmOut g (D8);
PwmOut b (D9);

// get the int value of the joystick state
int Thermostat::getErrorState() {
    if (this->m_error_state)
        return 1;
    else
        return 0;
}

// function to convert hue , saturation and  value to RGB - then update the RGB LED
// see http://en.wikipedia.org/wiki/HSL_and_HSV
void Thermostat::updateRGBLED(float H, float S, float V) {
    float f,h,p,q,t;
    int i;
    if( S == 0.0) {
        r = 1.0 - V;  // invert pwm !
        g = 1.0 - V;
        b = 1.0 - V;
        return;
    }
    if(H > 360.0) H = 0.0;   // check values
    if(S > 1.0) S = 1.0; 
    if(S < 0.0) S = 0.0;
    if(V > 1.0) V = 1.0;
    if(V < 0.0) V = 0.0;
    h = H / 60.0;
    i = (int) h;
    f = h - i;
    p = V * (1.0 - S);
    q = V * (1.0 - (S * f));
    t = V * (1.0 - (S * (1.0 - f)));

    switch(i) {
        case 0:
            r = 1.0 - V;  // invert pwm !
            g = 1.0 - t;
            b = 1.0 - p;
            break;
        case 1:
            r = 1.0 - q;
            g = 1.0 - V;
            b = 1.0 - p;
            break;
        case 2:
            r = 1.0 - p;
            g = 1.0 - V;
            b = 1.0 - t;
            break;
        case 3:
            r = 1.0 - p;
            g = 1.0 - q;
            b = 1.0 - V;
            break;
        case 4:
            r = 1.0 - t;
            g = 1.0 - p;
            b = 1.0 - V;
            break;
        case 5:
        default:
            r = 1.0 - V;
            g = 1.0 - p;
            b = 1.0 - q;
            break;
    }
}

// set all the LEDs
void Thermostat::setAllLEDs(int state) {
    led1.write(state);
    led2.write(state);
    led3.write(state);
    led4.write(state);
}

// set all the LEDs
void Thermostat::resetAllLEDs() {
    this->setAllLEDs(0);
}

// set all the LEDs
void Thermostat::blinkAllLEDs() {
    for(int i=0;i<4;++i) {
        wait(1.0);
        this->setAllLEDs(1);
        this->display("Blinking on...");
        wait(1.0);
        this->setAllLEDs(0);
        this->display("Blinking off...");
    }
}

// set the RGB LED color and brightness
void Thermostat::setRGBLED(double color, double bright) {
    // set the RGB LED value
    this->updateRGBLED(color,1.0,bright);
}

// turn the RGB LED Red
void Thermostat::turnRGBLEDRed() {
    this->m_rgbLEDColor = 0.0;
    this->m_rgbLEDBright = 0.2;
    this->setRGBLED(this->m_rgbLEDColor,this->m_rgbLEDBright); 
    this->m_error_state = true;
}

// turn the RGB LED Green
void Thermostat::turnRGBLEDGreen() {
    this->m_rgbLEDColor = 120.0;
    this->m_rgbLEDBright = 0.2;
    this->setRGBLED(this->m_rgbLEDColor,this->m_rgbLEDBright); 
    this->m_error_state = false;
}

// turn the RGB LED Blue (initializing state)
void Thermostat::turnRGBLEDBlue() {
    this->m_rgbLEDColor = 200.0;
    this->m_rgbLEDBright = 0.2;
    this->setRGBLED(this->m_rgbLEDColor,this->m_rgbLEDBright); 
    this->m_error_state = false;
}

// blink an LED
void Thermostat::blinkLED(DigitalOut led) {
    led = 1;
    wait(0.2);
    led = 0;
}

// blink the Transport TX LED
void Thermostat::blinkTransportTxLED() {
    this->blinkLED(led1);
}

// blink the Transport RX LED
void Thermostat::blinkTransportRxLED() {
    this->blinkLED(led2);
}


#endif // THERMOSTAT_LEDUTILS_H_