/* Copyright C2014 ARM, MIT License
 *
 * Author: Doug Anson (doug.anson@arm.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 #include "StatusReporter.h"
 
 // Logging
 #define LOG_CONSOLE(...) { if (this->logger() != NULL) { this->logger()->logConsole(__VA_ARGS__); } }
 
 // ****************** DreamForce 2014 ******************// 
 
 // TEMP SENSOR - The TMP36 sensor uses A3 in the shield.. See Definitions.h for the define.
 //
 // Uncommnent the following line to bind to the TMP36 temperature sensor
 //
 // AnalogIn tmp36_temp_sensor(TEMP_PIN);
 
 // ****************** DreamForce 2014 ******************// 
 
 StatusReporter::StatusReporter(HTTPClient *http,Logger *logger) : m_case_generator(http,logger), m_db(), m_rfid_reader(RFID_TX_PIN,RFID_RX_PIN) {
     this->m_logger = logger;
 }
 
 StatusReporter::~StatusReporter() {
 }
 
 Logger *StatusReporter::logger() { return this->m_logger; }
 
 // Calculate the ambient temperature of the TMP36 sensor in C...
 int StatusReporter::getLocalTemperature() {
     // TEMP SENSOR: temperature calculation per TMP36 data sheet
     //
     // Celcius temperature is calculated as follows (with 3.3V input):
     //
     //      tempC = ((temperature_sensor_voltage*3.3) - 0.6) * 100.0;
     //
     // You can use the following code snippet:
     // 
     //      float tempC = (float)(((float)tmp36_temp_sensor*3.3)-0.600)*100.0;
     //
     // in place of:
     // 
     //      float tempC = 0.0;
     // 
     // to calculate the ambient temperature via the TMP36 sensor (in Celcius)
     //
     
     float tempC = 0.0;
     
     // DEBUG
     LOG_CONSOLE("Ambient Temp: %.1f C",tempC);
     
     // convert to int for brevity...
     return (int)tempC;
 }
 
 void StatusReporter::checkAndReportOnStatus() { 
    // look for a readable RFID tag
    if(this->m_rfid_reader.readable()) {
        // capture the RFID id...
        LOG_CONSOLE("RFID: Found RFID.\r\nReading...");
        int rfid = this->m_rfid_reader.read();
        LOG_CONSOLE("RFID: ID %d found...\r\nProcessing...",rfid);
        
        // look it up in our ReportDB... proceed only if we find something we know about...
        char *name = this->m_db.lookupReportName(rfid);
        if (name != NULL) {
            // build out a simple subject for the case
            char subject[DB_MAX_NAME_LENGTH+1];
            memset(subject,0,DB_MAX_NAME_LENGTH+1);
            sprintf(subject,"%s case update",name);
            
            // create and dispatch a case
            this->m_logger->turnLEDPurple();
            char *description = this->m_db.lookupReportDescription(rfid);
            char *condition = this->m_db.lookupReportCondition(rfid);
            int temperature = this->getLocalTemperature();
            char *latitude = this->m_db.lookupReportLatitude(rfid);
            char *longitude = this->m_db.lookupReportLongitude(rfid);
            bool success = this->m_case_generator.createAnonymousCase(subject,description,condition,temperature,latitude,longitude);
            if (success == true) {
                LOG_CONSOLE("Case Generated!\r\nScanning...");
                this->m_logger->turnLEDGreen();
            }
            else {
                LOG_CONSOLE("Case Generation FAILED\r\nScanning...");
                this->m_logger->turnLEDYellow();
            }
        }
        else {
            // unrecognized RFID
            LOG_CONSOLE("RFID %d unknown.\r\nScanning...",rfid);
        }
     }
 }