/**
 * @file    SplitterAssembler.h
 * @brief   data buffer splitter and assembler header
 * @author  Doug Anson
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef __SPLITTER_ASSEMBLER_H__
#define __SPLITTER_ASSEMBLER_H__

#include "mbed.h"

// TUNABLES
#define MAX_FRAGMENTS           17      // maximum number of supported fragments... increase as needed... (max payload length = 340/DEF_FRAGMENT_LENGTH ~ 17)
#define DEF_FRAGMENT_LENGTH     20      // BLE: set to max MTU for BLE UART: defined by BLE_UART_SERVICE_MAX_DATA_LEN in UARTService.h

class SplitterAssembler {
    public:
        /**
        Constructor
        */
        SplitterAssembler();
        
        /**
        Split a data stream into "n" fragments of the given input fragement (from constructor) size
        @param data input input data buffer
        @param data_length input input data buffer length
        @returns number of fragments created for the input buffer
        */
        int split(uint8_t *data,int data_length);
        
        /**
        Get the ith fragment
        @param index input the ith index value
        @return the ith fragment pointer (null terminated)
        */
        uint8_t *get(int index);
        
        /** 
        Get the splitter fragment length
        @returns the configured splitter fragment length
        */
        int getSplitLength() { return DEF_FRAGMENT_LENGTH; }
        
        /**
        Get the number of fragments 
        @returns the number of fragments 
        */
        int getNumFragments() { return this->m_num_fragments; }
        
        /**
        Reset the Fragmenter/Assembler
        */
        void reset(void);
        
        /**
        Add a fragment
        @param fragment input fragment to add 
        @param fragment_length input the input fragment length (sanity checks...)
        @return the number of fragments currently stored
        */
        int add(uint8_t *fragment,int fragment_length);
        
        /**
        Assemble fragments
        @param buffer input the result buffer
        @param buffer_length input the result buffer length (maximum)
        @param reset_after_assemble input reset the Fragmenter/Assembler after assembly is complete (default: TRUE)
        @return the length of the assembled buffer
        */
        int assemble(uint8_t *buffer,int buffer_length,bool reset_after_assemble = true);
        
        /**
        Find a specific character in the collected fragments
        @param special_char the special character to look for
        @return true - special char is in one of the fragments, false, otherwise
        */
        bool hasCollectedCharacter(char special_char);
        
    private:
        int     calculateNumFragments(uint8_t *data,int data_length);
        int     calculateAssemblyLength(int buffer_length);
        void    dump(void);
        
        int     m_num_fragments;
        int     m_last_fragment_length;
        uint8_t m_fragments[MAX_FRAGMENTS][DEF_FRAGMENT_LENGTH+1];
};

#endif // __SPLITTER_ASSEMBLER_H__