/* Copyright C2014 ARM, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 #include "Logger.h"
 
 #if _UBLOX_PLATFORM    
    // Annunciations
    DigitalOut led1(P3_25);
    DigitalOut led2(P3_25);
    DigitalOut led3(P3_25);
    DigitalOut led4(P3_25);
    
    // Multi-color LED support
    PwmOut r(D5);
    PwmOut g(D9);
    PwmOut b(D8);  
#endif

#if _NXP_PLATFORM    
    // Annunciations
    DigitalOut led1(LED1);
    DigitalOut led2(LED2);
    DigitalOut led3(LED3);
    DigitalOut led4(LED4);  
    
    // Multi-color LED support
    PwmOut r(p23);
    PwmOut g(p24);
    PwmOut b(p25);  
#endif

#if _K64F_PLATFORM
    // Annunciations
    DigitalOut led1(LED1);
    DigitalOut led2(LED2);
    DigitalOut led3(LED1);
    DigitalOut led4(LED2);  
   
    // Multi-color LED support
    DigitalOut r(PTB22);
    DigitalOut g(PTE26);
    DigitalOut b(PTB21); 
#endif

// Memory statistics macro
#define LOGGER_MEM_STATS(x) \
    int s##x=0;\
    int *h##x = new int [1];\
    if (this->m_pc != NULL) this->m_pc->printf("\r\nMEMORY: stack: 0x%08x  heap: 0x%08x  avail: %d bytes\r\n", &s##x, h##x, &s##x-h##x);\
    if (h##x > &s##x)\
    error("collision\n");\
    delete [] h##x;\
    __nop();
    
 // constructor
 Logger::Logger(RawSerial *pc,LCDCLASS *lcd) {
     this->m_pc = pc;
     this->m_lcd = lcd;
     memset(this->m_message,0,MAX_LOG_MESSAGE+1);
     led1 = 0; led2 = 0; led3 = 0; led4 = 0;
 }

 // destructor
 Logger::~Logger() {
 }
 
 // VARARGS: log 
 void Logger::log(const char *format, ...)  { 
    // construct the log message
    memset(this->m_message,0,MAX_LOG_MESSAGE+1);
    va_list args;
    va_start(args, format);
    vsprintf(this->m_message, format, args);
    va_end(args);
    
    // make sure we have a message to log
    if (strlen(this->m_message) > 0) {
       // Log to serial...
       if (this->m_pc != NULL) {
            this->m_pc->printf(this->m_message);
            this->m_pc->printf("\r\n");
       }
        
       // Log to the LCD panel...
       if (this->m_lcd != NULL) {
            this->m_lcd->cls();
            this->m_lcd->locate(0,0);
            this->m_lcd->printf(this->m_message);
       }
    }
 } 
 
 // VARARGS: log 
 void Logger::logConsole(const char *format, ...)  { 
    // construct the log message
    memset(this->m_message,0,MAX_LOG_MESSAGE+1);
    va_list args;
    va_start(args, format);
    vsprintf(this->m_message, format, args);
    va_end(args);
    
    // make sure we have a message to log
    if (strlen(this->m_message) > 0) {
       // Log to serial...
       if (this->m_pc != NULL) {
            this->m_pc->printf(this->m_message);
            this->m_pc->printf("\r\n");
       }
    }
 } 
 
 // set the color LED 
 void Logger::setRGBLED(float H, float S, float V) {
    float f,h,p,q,t;
    int i;
    if( S == (float)0.0) {
        r = (float)1.0 - V;  // invert pwm !
        g = (float)1.0 - V;
        b = (float)1.0 - V;
        return;
    }
    if(H > (float)360.0) H = (float)0.0;   // check values
    if(S > (float)1.0) S = (float)1.0; 
    if(S < (float)0.0) S = (float)0.0;
    if(V > (float)1.0) V =(float) 1.0;
    if(V < (float)0.0) V = (float)0.0;
    h = (float)H / (float)60.0;
    i = (int) h;
    f = h - i;
    p = (float)V * ((float)1.0 - S);
    q = (float)V * ((float)1.0 - (S * f));
    t = (float)V * ((float)1.0 - (S * ((float)1.0 - f)));

    switch(i) {
        case 0:
            r = (float)1.0 - V;  // invert pwm !
            g = (float)1.0 - t;
            b = (float)1.0 - p;
            break;
        case 1:
            r = (float)1.0 - q;
            g = (float)1.0 - V;
            b = (float)1.0 - p;
            break;
        case 2:
            r = (float)1.0 - p;
            g = (float)1.0 - V;
            b = (float)1.0 - t;
            break;
        case 3:
            r = (float)1.0 - p;
            g = (float)1.0 - q;
            b = (float)1.0 - V;
            break;
        case 4:
            r = (float)1.0 - t;
            g = (float)1.0 - p;
            b = (float)1.0 - V;
            break;
        case 5:
        default:
            r = (float)1.0 - V;
            g = (float)1.0 - p;
            b = (float)1.0 - q;
            break;
    }
 }

 // turn the RGB LED specific colors
 void Logger::turnLEDRed() { this->setRGBLED(0.0,1.0,0.2); }
 void Logger::turnLEDGreen() { this->setRGBLED(120.0,1.0,0.2); }
 void Logger::turnLEDBlue() { this->setRGBLED(200.0,1.0,0.2); }
 void Logger::turnLEDPurple() { this->setRGBLED(261.9,1.0,0.2); }
 void Logger::turnLEDOrange() { this->setRGBLED(51.0,1.0,0.2); }
 void Logger::turnLEDBlack() { this->setRGBLED(0,0,0); }
 void Logger::turnLEDYellow() { this->setRGBLED(60.0,1.0,0.133); }
 
 void Logger::blinkTransportTxLED() { this->blinkLED(led1); }
 void Logger::blinkTransportRxLED() { this->blinkLED(led2); }

 // blink an LED
 void Logger::blinkLED(DigitalOut led) {
    led = 1;
    wait_ms(BLINK_TIME);
    led = 0;
 }
 