/* Copyright C2014 ARM, MIT License
 *
 * Author: Doug Anson (doug.anson@arm.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files the "Software", to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
  
 // our Serial port
 #include "BufferedSerial.h"
 BufferedSerial pc(USBTX, USBRX);
 
 // Ethernet
 #include "EthernetInterface.h"
 EthernetInterface ethernet;
 
 // HTTP 
 #include "HTTPClient.h"
 HTTPClient http;
 
 // StatusReporter
 #include "StatusReporter.h"
 
 #if _K64F_PLATFORM
 // the K64F does not have a unique MAC address - so we must give it one... 
 extern "C" void mbed_mac_address(char *mac)
 {
    // Fetch word 0
    uint32_t word0 = *(uint32_t *)0x40048060;
    // Fetch word 1
    // we only want bottom 16 bits of word1 (MAC bits 32-47)
    // and bit 9 forced to 1, bit 8 forced to 0
    // Locally administered MAC, reduced conflicts
    // http://en.wikipedia.org/wiki/MAC_address
    uint32_t word1 = *(uint32_t *)0x4004805C;
    word1 |= 0x00000200;
    word1 &= 0x0000FEFF;
    
    mac[0] = (word1 & 0x000000ff);
    mac[1] = (word1 & 0x0000ff00) >> 8;
    mac[2] = (word0 & 0xff000000) >> 24;
    mac[3] = (word0 & 0x00ff0000) >> 16;
    mac[4] = (word0 & 0x0000ff00) >> 8;
    mac[5] = (word0 & 0x000000ff);
 }
 #endif // _K64F_PLATFORM
  
 // Main Task...
 void mainTask(void const *v) {
    // create our object instances 
    Logger logger(&pc,NULL);
    
    // announce
    logger.log("ARM/DreamForce 2014 mbed Status Reporter v%s",APP_VERSION);
    logger.turnLEDBlue();
    
    // initialize Ethernet
    logger.log("Initializing Ethernet...");
    ethernet.init();
    
    // get a DHCP address and bring the network interface up
    logger.log("Getting IP Address...");
    logger.turnLEDOrange();
    if (ethernet.connect() == 0) {
        // log our IP address (DHCP)
        logger.log("IP Address: %s",ethernet.getIPAddress());
        
        // create the StatusReporter
        StatusReporter status_reporter(&http,&logger);
        
        // entering main loop
        logger.log("Entering Main Loop...\r\nScanning...");
        logger.turnLEDGreen();
        
        // Enter the main loop
        while(true) {
             // check and report on status updates
             status_reporter.checkAndReportOnStatus();
        }
     }
     else {
         logger.log("No Network... Exiting...");
         logger.turnLEDRed();
         exit(1);
     }
     
     // disconnect
     logger.log("Disconnecting...");
     logger.turnLEDOrange();
     ethernet.disconnect();
     
     // Exit
     logger.log("Exiting...");
     logger.turnLEDBlue();
     exit(1);
  }
  
  // main entry
  int main() {
     Thread workerTask(mainTask, NULL, osPriorityNormal, STACK_SIZE);
     while (true) {
        Thread::wait(10*WAIT_TIME_MS);
     }
  }