#include "mbed.h"               // Pour accès aux différentes librairies du Mbed
#include "us_ticker_api.h"      // Accès à un compteur 32 bits en "free running"
#include "mlcd_32pt.h"          // Prototypes des fonctions et constantes pour l'afficheur µLCD32

#define LCD_TX      p28         // Broches pour communication avec l'afficheur µLCD32
#define LCD_RX      p27
#define LCD_RAZ     p30         // Broche pour remise à zéro de l'afficheur µLCD32
#define DELAI_RAZ   2000        // Constante pour délai en msec pour la remise à zéro de l'affichewur µLCD32
#define DELAI_CLIGNOTE 25       // Constante pour délai en msec du clignotement de l'ACK


Serial lcd(LCD_TX, LCD_RX);     // Objet pour communication série avec l'afficheur µLCD32
Serial term(USBTX, USBRX);      // Terminal série vers le PC pour mise au point
DigitalOut led_ack(LED1);       // Sortie vers LED1 pour indiquer si il y a réception de l'ACK de l'afficheur µLCD32 (2 clignotements rapides : ACK reçu, LED1 allumé : aucun ACK)
DigitalOut sig_raz(LCD_RAZ);    // Sortie pour contrôle de la remise à zéro de l'afficheur µLCD32
int reponse;

/*
    Remise à zéro de l'afficheur

*/
void raz(void)
{
   sig_raz = 0;         // Impulsion négative pour raz de l'afficheur
   wait_ms(25);
   sig_raz = 1;
   wait_ms(DELAI_RAZ);  // Délai d'attente après une remise à zéro
}

/* 
    Clignotement de la del pour l'ACK
    
    time : durée du clignotement en msec.  
*/
void pulse(int time)
{
    int i;
    for(i=0;i<2;i++)   // Clignote 2 fois
    {
      led_ack = 1;     // Clignotement de la LED de "time" msec.
      wait_ms(time);
      led_ack = 0;
      wait_ms(time);
    }
}

/*
    Lecture de l'ACK de l'afficheur LCD
*/

int ack(void)
{
   int rep = NAK;

    if ((rep = lcd.getc()) != ACK)  // Lecture de l'ACK
    {
        led_ack = 1;       // Allume le LED en cas d'erreur
        rep = NAK;
    }
    else
        pulse(DELAI_CLIGNOTE); // Indique que l'ACK est reçu
   
    wait_ms(25);

    return rep;
}

/*
    Conversion d'une couleur sur 24 bits en une couleur sur 16 bits pour l'afficheur µLCD32
    
    r : Couleur rouge sur 8 bits
    g : Couleur verte sur 8 bits
    b : Couleur bleu sur 8 bits
    
    Valeur de sortie : Couleur résultante sur 16 bits pour l'afficheur µLCD32
*/
int rgb(char r, char g, char b)
{
    return (r>> 3 << 11) | (g>> 2 << 5) | (b>> 3);  // Conversion des la valeur sur 24 bits en 16 bits
}

/*
    Dessin d'un bouton de contrôle pour l'afficheur µLCD32
    
    state : État du bouton ( 0 : pressé, 1 : relaché)
    x, y : Position du bouton
    bgcolor : Couleur de fond du bouton sur 16 bits
    font : Police des caractères (0 : 5x7, 1 : 8x8, 2 : 8x12, 3 : 12x16
    fgcolor : Couleur de la police sur 16 bits
    width, height : Facteur multiplicatif pour l'affichage des caractères
    *str : Pointeur vers chaîne de caractères à afficher dans le bouton
    
    Valeur de sortie : ACK ou NAK

*/
int drawTextButton(char state, int x, int y, int bgcolor,char font, int fgcolor, int width, int height, unsigned char *str)
{
    lcd.putc(BUTTON_TEXT);         // Commande pour bouton texte
    lcd.putc(state);               // Paramètres de la commande
    lcd.putc(x>>8);
    lcd.putc(x&0xff);
    lcd.putc(y>>8);
    lcd.putc(y&0xff);
    lcd.putc(bgcolor>>8);
    lcd.putc(bgcolor&0xff);
    lcd.putc(font);
    lcd.putc(fgcolor>>8);
    lcd.putc(fgcolor&0xff);
    lcd.putc(width);
    lcd.putc(height);
    while(*str)                     // Chaîne de caractères à afficher
        lcd.putc(*str++);

    lcd.putc(0);                    // Fin de la chaîne

    return ack();                   // Retourne la lecture de l'ACK
}

/*
    Dessin d'une chaîne de caractères pour l'afficheur µLCD32
     
    x, y : Position du début de la chaîne de caractères
    font : Police des caractères (0 : 5x7, 1 : 8x8, 2 : 8x12, 3 : 12x16
    fgcolor : Couleur de la police sur 16 bits
    *str : Pointeur vers chaîne de caractères à afficher dans le bouton
    
    Valeur de sortie : ACK ou NAK
    
*/

int drawString(int x, int y, char font, int fgcolor, int width, int height, unsigned char *str)
{
    lcd.putc(STRING);             // Commande pour le dessin d'une chaîne de caractères sur l'écran µLCD32
    lcd.putc(x>>8);               // Paramèrtre de la commande
    lcd.putc(x&0xff);
    lcd.putc(y>>8);
    lcd.putc(y&0xff);
    lcd.putc(font);
    lcd.putc(fgcolor>>8);
    lcd.putc(fgcolor&0xff);
    lcd.putc(width);
    lcd.putc(height);

    while(*str)                  // Chaîne de caractères à afficher
        lcd.putc(*str++);        

    lcd.putc(0);                 // Fin de la chaîne

    return ack();
}

/*
    Change la couleur de fond de l'afficheur µLCD32
    
    fgcolor : Couleur de fond sur 16 bits
    
    Valeur de sortie : ACK ou NAK

*/
int setbgColor(int fgcolor)
{
    lcd.putc(BGCOLOR);
    lcd.putc(fgcolor>>8);
    lcd.putc(fgcolor&0xff);
    return ack();
}

/*
    Initialisation du menu (ne fait rien à part d'effacer l'écran µLCD32)

    Valeur de sortie : ACK ou NAK
*/

int initMenu()
{
    int rep;

    lcd.putc(ERASE);    // Commande pour effacer l'écran
    rep = ack();        // Lecture de l'ACK
 
    return rep;
}

/*
    Initialise la vitesse de communication série avec l'afficheur
    
    baudrate : Vitesse de communication
    
    Valeur de sortie : ACK ou NAK

*/

int init_com(int baudrate)
{
    
    lcd.putc(SET_BAUD); // Commande pour changer de vitesse de communication
    lcd.baud(baudrate); // Nouvelle vitesse     
    
    return 1;//ack();       // Lecture de l'ACK
}

/*
    Initialisation de l'afficheur µLCD32
*/

void init_lcd(void)
{
    raz();  // remise à zéro de l'afficheur µLCD32
   
    
    lcd.putc(AUTO_BAUD); //Baudrate automatique
    ack();
    wait_ms(20);
        
    // Active l'écran tactile du µLCD32
    lcd.putc('Y');
    lcd.putc(0x05);
    lcd.putc(0x00);
    ack();
    
    // Active toute la surface de l'écran tactile
    lcd.putc('Y');
    lcd.putc(0x05);
    lcd.putc(0x02);
    ack();
    
    // Initialise le menu
    initMenu();
    
    //Initialise le GET TOUCH
    lcd.putc('o');
    lcd.putc(0x04);
    return reponse;
    
}

int double_clic(void)
{
    

    if (reponse == 2)
    {
        lcd.putc('w');
        lcd.putc(0x01);
        lcd.putc(0xF4);
        reponse = ack();
        
        if (reponse == 6)
        {
            lcd.putc('E');
            return ack();
        }
     }
}

int copier_coller(void)
{
    lcd.putc('c');
    lcd.putc(0x00);
    lcd.putc(0x00);
    lcd.putc(0x00);
    lcd.putc(0x0F);
    lcd.putc(0x00);
    lcd.putc(0x00);
    lcd.putc(0x00);
    lcd.putc(0x00);
    lcd.putc(0x01);
    lcd.putc(0x3F);
    lcd.putc(0x00);
    lcd.putc(0xE0);
    return ack();
    
}
