#include "mbed.h"

#define USE_CAP 1
#define USE_WIFI_BASIC 0
#define USE_WIFI_ADVANCED 1

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);
Serial pc(USBTX, USBRX); // tx, rx

#if USE_CAP
#include <mpr121.h>
// Create the interrupt receiver object on pin 26
InterruptIn interrupt(p26);
// Setup the i2c bus on pins 9 and 10
I2C i2c(p9, p10);
// Setup the Mpr121:
// constructor(i2c object, i2c address of the mpr121)
Mpr121 mpr121(&i2c, Mpr121::ADD_VSS);

void fallInterrupt();
#endif

// WiFi stuff
#if USE_WIFI_BASIC
Serial esp(p28, p27); // tx, rx
DigitalOut reset(p29);
Timer t;

char ssid[32] = "GTother";     // enter WiFi router ssid inside the quotes
char pwd [32] = "GeorgeP@1927"; // enter WiFi router password inside the quotes
int  count,ended,timeout;
char buf[1024];
char snd[255];

void SendCMD(), getreply(), ESPconfig(), ESPsetbaudrate();

void basicWifiSetup();
void basicWifiLoop();
#endif

#if USE_WIFI_ADVANCED
#include "HTTPClient.h"
#include "ESP8266Interface.h"
#include "TCPSocketConnection.h"
#include "Websocket.h"
#include "rtos.h"

const char *httpDest = "http://143.215.98.59/";
ESP8266Interface wifi(p28, p27, p29, "GTother", "GeorgeP@1927", 115200); // TX,RX,Reset,SSID,Password,Baud

HTTPClient http;
char responseBuffer[512];

void advWifiSetup();
void advWifiTest();
void advWifiSendCommand(int val);

Mutex advWifiMutex;
void advWifiThread(void const *args);
int gKeycode = -1;
bool gKeycodeLatch = false;
#endif

int main() {
    pc.printf("====\n\r===\n\r===\n\r=== Reset ====\n\r====\n\r\n\r");
    
    led1 = led2 = led3 = led4 = 0;
    
    #if USE_CAP
    interrupt.fall(&fallInterrupt);
    interrupt.mode(PullUp);
    #endif

    // Wifi setup
    #if USE_WIFI_BASIC
    basicWifiSetup();
    #endif
    
    #if USE_WIFI_ADVANCED
    advWifiSetup();
    Thread t2(advWifiThread);
    #endif

    while(1) {
        #if USE_WIFI_BASIC
        basicWifiLoop();
        #endif
        
        #if USE_WIFI_ADVANCED
        //for (int i = 0; i < 2; i++) {
        //    advWifiSendCommand(i);
        //}
        #endif
    }
}

#if USE_CAP
// Light up LEDs with capacitive touch interrupt stuff
void fallInterrupt() {
    int key_code = 0;
    int i = 0;
    
    // Address 0x00 has touch status of 0 - 7 (7, 6, 5, 4, 3, 2, 1, 0)
    // Address 0x01 has touch status of 8 - 11 (X,X,X,X, 11, 10, 9, 8)
    // Combined value is a bitfield type thing
    int value = mpr121.read(0x00);
    value += mpr121.read(0x01) << 8;
    
    // Takes highest set key and uses it
    int bits[12];
    for (i = 0; i < 12; i++) {
        bits[i] = (value >> i) & 0x01;
        if (bits[i] == 1) {
            key_code = i + 1;
        }
    }
    led4 = (key_code >> 0) & 0x01;
    led3 = (key_code >> 1) & 0x01;
    led2 = (key_code >> 2) & 0x01;
    led1 = (key_code >> 3) & 0x01;

    #if USE_WIFI_ADVANCED
    gKeycode = key_code - 1;
    gKeycodeLatch = true;
    #endif
}
#endif

#if USE_WIFI_BASIC
// Sets new ESP8266 baurate, change the esp.baud(xxxxx) to match your new setting once this has been executed
void ESPsetbaudrate()
{
    strcpy(snd, "AT+CIOBAUD=115200\r\n");   // change the numeric value to the required baudrate
    SendCMD();
}
 
//  +++++++++++++++++++++++++++++++++ This is for ESP8266 config only, run this once to set up the ESP8266 +++++++++++++++
void ESPconfig()
{
    wait(5);
    strcpy(snd,"AT\r\n");
    SendCMD();
    wait(1);
    strcpy(snd,"AT\r\n");
    SendCMD();
    wait(1);
    strcpy(snd,"AT\r\n");
    SendCMD();
    timeout=1;
    getreply();
    wait(1);
    pc.printf("\f---------- Starting ESP Config ----------\r\n\n");
 
    pc.printf("---------- Reset & get Firmware ----------\r\n");
    strcpy(snd,"AT+RST\r\n");
    SendCMD();
    timeout=5;
    getreply();
    pc.printf(buf);
 
    wait(2);
 
    pc.printf("\n---------- Get Version ----------\r\n");
    strcpy(snd,"AT+GMR\r\n");
    SendCMD();
    timeout=4;
    getreply();
    pc.printf(buf);
 
    wait(3);
 
    // set CWMODE to 1=Station,2=AP,3=BOTH, default mode 1 (Station)
    pc.printf("\n---------- Setting Mode ----------\r\n");
    strcpy(snd, "AT+CWMODE=1\r\n");
    SendCMD();
    timeout=4;
    getreply();
    pc.printf(buf);
 
    wait(2);
 
    // set CIPMUX to 0=Single,1=Multi
    pc.printf("\n---------- Setting Connection Mode ----------\r\n");
    strcpy(snd, "AT+CIPMUX=1\r\n");
    SendCMD();
    timeout=4;
    getreply();
    pc.printf(buf);
 
    wait(2);
 
    pc.printf("\n---------- Listing Access Points ----------\r\n");
    strcpy(snd, "AT+CWLAP\r\n");
    SendCMD();
    timeout=15;
    getreply();
    pc.printf(buf);
 
    wait(2);
 
    pc.printf("\n---------- Connecting to AP ----------\r\n");
    pc.printf("ssid = %s   pwd = %s\r\n",ssid,pwd);
    strcpy(snd, "AT+CWJAP=\"");
    strcat(snd, ssid);
    strcat(snd, "\",\"");
    strcat(snd, pwd);
    strcat(snd, "\"\r\n");
    SendCMD();
    timeout=10;
    getreply();
    pc.printf(buf);
 
    wait(5);
 
    pc.printf("\n---------- Get IP's ----------\r\n");
    strcpy(snd, "AT+CIFSR\r\n");
    SendCMD();
    timeout=3;
    getreply();
    pc.printf(buf);
 
    wait(1);
 
    pc.printf("\n---------- Get Connection Status ----------\r\n");
    strcpy(snd, "AT+CIPSTATUS\r\n");
    SendCMD();
    timeout=5;
    getreply();
    pc.printf(buf);
 
    pc.printf("\n\n\n  If you get a valid (non zero) IP, ESP8266 has been set up.\r\n");
    pc.printf("  Run this if you want to reconfig the ESP8266 at any time.\r\n");
    pc.printf("  It saves the SSID and password settings internally\r\n");
    wait(10);
}
 
void SendCMD()
{
    esp.printf("%s", snd);
}
 
void getreply()
{
    memset(buf, '\0', sizeof(buf));
    t.start();
    ended=0;
    count=0;
    while(!ended) {
        if(esp.readable()) {
            buf[count] = esp.getc();
            count++;
        }
        if(t.read() > timeout) {
            ended = 1;
            t.stop();
            t.reset();
        }
    }
}

// Basic wifi setup code
void basicWifiSetup() {
    reset=0; //hardware reset for 8266
    pc.baud(9600);  // set what you want here depending on your terminal program speed
    pc.printf("\f\n\r-------------ESP8266 Hardware Reset-------------\n\r");
    wait(0.5);
    reset=1;
    timeout=2;
    getreply();
 
    esp.baud(115200);   // change this to the new ESP8266 baudrate if it is changed at any time.
 
    //ESPsetbaudrate();   //******************  include this routine to set a different ESP8266 baudrate  ******************
 
    ESPconfig();        //******************  include Config to set the ESP8266 configuration  ***********************
}

// Loop for basic wifi stuff
void basicWifiLoop() {
    pc.printf("\n---------- Listing Acces Points ----------\r\n");
    strcpy(snd, "AT+CWLAP\r\n");
    SendCMD();
    timeout=15;
    getreply();
    pc.printf(buf);
    wait(2);
    pc.printf("\n---------- Get IP and MAC ----------\r\n");
    strcpy(snd, "AT+CIFSR\r\n");
    SendCMD();
    timeout=10;
    getreply();
    pc.printf(buf);
    wait(2);
}
#endif

#if USE_WIFI_ADVANCED
// Setup advanced wifi
void advWifiSetup() {
    pc.baud(9600);
    wifi.init(); //Reset
    wifi.connect(); //Use DHCP
    
    advWifiTest(); // Test the connection
}

// Test that the advanced wifi is working with pastebin get/post
void advWifiTest() {
    bool getSuccess = false,
         postSuccess = false;
    
    //GET
    pc.printf("\nTrying to fetch page using GET...\n\r");
    int httpResp = http.get("http://54.175.222.246/get", responseBuffer, 512);//IP address is httpbin.org/get
    if (!httpResp) {
        pc.printf("Page fetched successfully - read %d characters\n\r", strlen(responseBuffer));
        pc.printf("Result: %s\n\r", responseBuffer);
        getSuccess = true;
    } else {
        pc.printf("Error - ret = %d - HTTP return code = %d\n\r", httpResp, http.getHTTPResponseCode());
        getSuccess = false;
    }
    
    //POST
    HTTPMap postDataMap;
    HTTPText responseTextBind(responseBuffer, 512);
    postDataMap.put("Hello", "World");
    postDataMap.put("test", "1234");
    pc.printf("\nTrying to POST data to httpbin.org/post...\n\r");
    httpResp = http.post("http://54.175.222.246/post", postDataMap, &responseTextBind);//IP address is httpbin.org/post
    if (!httpResp) {
      pc.printf("Executed POST successfully - read %d characters\n\r", strlen(responseBuffer));
      pc.printf("Result: %s\n\r", responseBuffer);
      postSuccess = true;
    }
    else {
      pc.printf("Error - ret = %d - HTTP return code = %d\n\r", httpResp, http.getHTTPResponseCode());
      postSuccess = false;
    }
    
    if (getSuccess && postSuccess) {
        pc.printf("\n\r=== WIFI SETUP AND GOOD TO GO====\n\r\n\r");
    } else {
        pc.printf("\n\r=== WIFI BAD - Try at your own risk... ====\n\r\n\r");
    }
}

// Send an advanced wifi command (non-blocking)
// INPUTS:
//    keycode - Keycode from capacitive touch screen
void advWifiSendCommand(int keycode) {
    pc.printf("Send command %d\n\r", keycode);
    
    char sendVal[32];
    switch (keycode) {
        case 2:
            sprintf(sendVal, "fwd");
            break;
        case 5:
            sprintf(sendVal, "left");
            break;
        case 7:
            sprintf(sendVal, "right");
            break;
        case 10:
            sprintf(sendVal, "back");
            break;
        default:
            pc.printf("\tBailing from bad keycode '%d'...\n\r", keycode);
            return; // Bail
    }
    
    // We good to go? Grab a mutex
    //if (!advWifiMutex.trylock()) {
    //    pc.printf("\tFailed to lock mutex. Bailing...\n\r");
    //    return;
    //}
    
    // Clear buffer
    responseBuffer[0] = '\0';
    
    // Setup bindings
    HTTPMap postDataMap;
    HTTPText responseTextBind(responseBuffer, 512);
    postDataMap.put("keycode", sendVal);
    
    pc.printf("Turning: %s with %d\t(%s)\n\r", sendVal, keycode, httpDest);
    int httpResp = http.post(httpDest, postDataMap, &responseTextBind);
    if (!httpResp) {
        pc.printf("\tSuccess!\n\r");
        pc.printf("\tResult: %s\n\r", responseBuffer);
    } else {
        pc.printf("\tFailed!\n\r");
        pc.printf("\tResult: %s\n\r", responseBuffer);
        pc.printf("\tRespVal(%d) \ ResponseCode(%d)\n\r", httpResp, http.getHTTPResponseCode());
    }
    
    //advWifiMutex.unlock();
}

void advWifiThread(void const *args) {
    while(1) {
        if (gKeycodeLatch && gKeycode != -1) {
            advWifiSendCommand(gKeycode);
            gKeycodeLatch = false;
        }
        
        Thread::wait(100);
    }
}
#endif