 /*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#include "Timer64Simple.h"
#include "ticker_api.h"
#include "us_ticker_api.h"

Timer64Simple::Timer64Simple() :
    _timerRunning(false),
    _tickerLastTimeUsec(0u), 
    _totalTimeUsec(TIMER64_TOTAL_TIME_INIT), 
    _ticker_data(get_us_ticker_data())
{
    ;
}

Timer64Simple::~Timer64Simple()
{
    ;
}

void Timer64Simple::start(void)
{
    if (!_timerRunning)
    {
        _tickerLastTimeUsec = ticker_read(_ticker_data);
        _timerRunning = true;
    }
}

void Timer64Simple::stop(void)
{
    if (_timerRunning)
    {
        _read_us();
        _timerRunning = false;
    }
}

void Timer64Simple::reset(void)
{
    if (_timerRunning)
    {
        _tickerLastTimeUsec = ticker_read(_ticker_data);
    }
    
    _totalTimeUsec = TIMER64_TOTAL_TIME_INIT;
}

uint64_t Timer64Simple::read_us()
{
    if (_timerRunning)
    {
        return _read_us();
    }
    return _totalTimeUsec;
}

uint64_t Timer64Simple::read_ms()
{
    if (_timerRunning)
    {
        return _read_us()/1000LU;
    }
    return _totalTimeUsec/1000LU;
}

double Timer64Simple::read()
{
    if (_timerRunning)
    {
        return (double)_read_us()/1000000.0L;
    }
    return (double)_totalTimeUsec/1000000.0L;
}

bool Timer64Simple::is_running()
{
    return(_timerRunning);
}

uint64_t Timer64Simple::_read_us()
{
    timestamp_t tickerCurrentTimeUsec = ticker_read(_ticker_data);
    
    // check for ticker time rollover
    
    if (tickerCurrentTimeUsec >= _tickerLastTimeUsec)
    {
        _totalTimeUsec += (uint64_t)(tickerCurrentTimeUsec - _tickerLastTimeUsec);
    }
    else
    {
        // rollover!
        _totalTimeUsec += (uint64_t)(4294967296U - _tickerLastTimeUsec) + (uint64_t)tickerCurrentTimeUsec;
        
    }
    
    _tickerLastTimeUsec = tickerCurrentTimeUsec;
    return(_totalTimeUsec);
}
