/* mbed Microcontroller Library
 * Copyright (c) 2006-2016 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Modified by Tom Doyle, 2016
 *             <ThomasEDoyle@gmail.com>
 *
 * Modified by Andriy Makukha, 2017
 *             <andriy@mzjtechnology.com>
 *             Shenzhen MZJ Technology, Co.
 */
 
#ifndef __TIMER64SIMPLE__
#define __TIMER64SIMPLE__

#include "platform.h"
#include "ticker_api.h"
#include <stdint.h>

#define TIMER64_TOTAL_TIME_INIT                     0llu

/** Timer64Simple class
 *  Used to define a 64 bit timer that is NOT thread safe.  The Timer64Simple
 *  behaves in a similiar fashion to the mbed Timer class with the most notable
 *  exception that the Timer64Simple will not rollover after 2^31 microseconds
 *  (approximately 35.8 minutes). The Timer64Simple extends the rollover time
 *  to 2^64 microseconds (more than 584,000 years!)
 *
 *  Unlike the Timer64 class by Tom Doyle, Timer64Simple class is simplified to
 *  be NOT thread safe. It is also assumed that read(), read_ms() or read_us()
 *  functions are called at least once per 17 minutes (preferably faster!) 
 *  while the timer is running (since this class lacks internal rollover check 
 *  timer). In fact, Timer64Simple is simplifed to the point where it's almost 
 *  trivial.
 *
 *  If you need smarter version, check out the original code here:
 *      https://developer.mbed.org/users/tedoyle/code/Timer64/
 *
 *  Demo program:
 *  @code
 *  #include "mbed.h"
 *  #include "Timer64Simple.h"
 *
 *  Timer64Simple timer64;
 *
 *  int main()
 *  {
 *      timer64.start();
 *      wait_ms(100);
 *      timer64.stop();
 *      wait_ms(100);
 *      printf("Test - Elapsed time = %llu usec\r\n", timer64.read_us());   // %llu -- unsigned long long int
 *       
 *      timer64.reset();
 *      timer64.start();
 *      while (1)
 *      {
 *          wait(60);           // wait 1 minute
 *          printf("Time = %llu ms\r\n", timer64.read_ms());
 *      }
 *  }
 *  @endcode
*/

class Timer64Simple
{

public:
    Timer64Simple();
    ~Timer64Simple();

    /** Start the timer */
    void start(void);

    /** Stop the timer  */
    void stop(void);

    /** Reset the timer */
    void reset(void);

    /** Read the timer value in microseconds 
     *  
     *  @returns
     *      timeInUsec - current time in microseconds
     */
    uint64_t read_us(void);

    /** Read the timer value in milliseconds
     *  
     *  @returns
     *      timeInMsec - current time in milliseconds
     */
    uint64_t read_ms(void);

    /** Read the timer value in seconds
     *  
     *  @returns
     *      timeInMsec - current time in seconds
     */
    double read(void);
    
    /** Check to see if timer is running
     *  
     *  @returns
     *      boolean running status
     */
    bool is_running();

private:
    bool _timerRunning;                 // whether the timer is running
    timestamp_t _tickerLastTimeUsec;    // the start time of the latest slice
    uint64_t _totalTimeUsec;            // any accumulated time from previous slices
    const ticker_data_t *_ticker_data;
    
    uint64_t _read_us(void);
};

#endif  //__TIMER64SIMPLE__
