#ifndef STEERMOTOR_H
#define STEERMOTOR_H

#include "mbed.h"

class SteerMotor {
private:
    DigitalOut _dir;
    DigitalOut _step;
    float _curr_angle;
    float _setpoint_angle;
    uint32_t _prev_time;


public:
    SteerMotor(DigitalOut dir, DigitalOut step) : _dir(dir), _step(step), _setpoint_angle(0), _prev_time(0) {}
    
    void set_angle(float angle) { _setpoint_angle = angle; }
    float get_angle() { return _curr_angle; }
    void zero() { _curr_angle = 0; }
    void kill() { _setpoint_angle = _curr_angle; }
    bool is_on_target(){
        return fabs(_curr_angle - _setpoint_angle) < 1.0f ||
               fabs(_curr_angle - _setpoint_angle + 360.0f) < 1.0f ||
               fabs(_curr_angle - _setpoint_angle - 360.0f) < 1.0f;
    }
    
    void update(uint32_t curr_time) {
        // Check if time has elapsed to update (100 Hz)
        if ((curr_time - _prev_time) < 5000) return;
        _prev_time = curr_time;
        
        // If step is high make it go low
        if (_step) {_step = 0; return;}
        
        if (is_on_target()) return;
        
        // Calculate the direction to move the stepper motor
        if (_curr_angle < _setpoint_angle && _curr_angle > _setpoint_angle - 180.0f) _dir = 1;
        else if (_curr_angle < _setpoint_angle) _dir = 0;
        else if (_curr_angle > _setpoint_angle && _curr_angle < _setpoint_angle + 180.0f) _dir = 0;
        else _dir = 1;
        
        // Take step
        _step = 1;
        
        // Calculate new current angle
        _curr_angle += (1.8f * 20.0f / 50.0f * (_dir ? 1 : -1));
        if (_curr_angle >= 360.0f) _curr_angle -= 360.0f;
        if (_curr_angle < 0.0f) _curr_angle += 360.0f;
    }
    
    
};

#endif