#include <stdint.h>
#include <stdlib.h>

#include "i2c1.h"
#include "mstimer.h"
#include "clk.h"
#include "tm.h"

// commands
#define LCD_CLEARDISPLAY 0x01
#define LCD_DISPLAYCONTROL 0x08
#define LCD_FUNCTIONSET 0x20

// flags for display on/off control
#define LCD_DISPLAYON 0x04
#define LCD_DISPLAYOFF 0x00

// flag for entry mode
#define LCD_ENTRYLEFT 0x02

// flags for function set
#define LCD_8BITMODE 0x10
#define LCD_2LINE 0x08
#define LCD_5x10DOTS 0x04

static uint8_t data[2]; //This must be static as it is used in the main routine
static int _result = I2C1_RESULT_NONE;

static void send(int length)
{
    I2C1Send(0x3e, length, data, &_result);
}

static void command(uint8_t cmd)
{
    data[0] = 0x80;
    data[1] = cmd;
    send(2);
}
void write(uint8_t value)
{
    data[0] = 0x40;
    data[1] = value;
    send(2);
}

#define TODO_NONE           0
#define TODO_INIT_START     1
#define TODO_INIT_FUNCTION  2
#define TODO_INIT_CONTROL   3
#define TODO_INIT_CLEAR     4
#define TODO_READY          5
#define TODO_SEND_START     6
#define TODO_SEND_CURSOR0   7
#define TODO_SEND_TEXT0     8
#define TODO_SEND_CURSOR1   9
#define TODO_SEND_TEXT1    10

static int _todo = TODO_NONE;

static uint32_t _MsTimerWaitAfter = 0;
static int      _waitAfterMs      = 0;

static const char* _pText0 = 0;
static const char* _pText1 = 0;
static int i = 0;

static void LcdSendText(char* pText0, char* pText1)
{
    _pText0 = pText0;
    _pText1 = pText1;
    _todo = TODO_SEND_START;
}

void LcdInit()
{
    _todo = TODO_INIT_START;
}

static void handleDoIt()
{
    if (I2C1Busy) return;
    if (!MsTimerRelative(_MsTimerWaitAfter, _waitAfterMs)) return;
    _waitAfterMs = 0; //Don't wait unless specified below
    _MsTimerWaitAfter = MsTimerCount;
    switch (_todo)
    {
        case TODO_INIT_START:
            _todo = TODO_INIT_FUNCTION;
            _waitAfterMs = 15;
            break;
        case TODO_INIT_FUNCTION:
            command(LCD_FUNCTIONSET | LCD_2LINE | LCD_5x10DOTS); //2 line
            _todo = TODO_INIT_CONTROL;
            _waitAfterMs = 2;
            break;
        case TODO_INIT_CONTROL:
            command(LCD_DISPLAYCONTROL | LCD_DISPLAYON);
            _todo = TODO_INIT_CLEAR;
            _waitAfterMs = 2;
            break;
        case TODO_INIT_CLEAR:
            command(LCD_CLEARDISPLAY);
            _todo = TODO_READY;
            _waitAfterMs = 15;
            break;
        case TODO_READY:
            break;
        case TODO_SEND_START:
            _todo = TODO_SEND_CURSOR0;
            break;
        case TODO_SEND_CURSOR0:
            command(0x80);
            _todo = TODO_SEND_TEXT0;
            i = 0;
            _waitAfterMs = 10;
            break;
        case TODO_SEND_TEXT0:
            if (_pText0 && _pText0[i]) write(_pText0[i++]);
            else                       _todo = TODO_SEND_CURSOR1;
            _waitAfterMs = 5;
            break;
        case TODO_SEND_CURSOR1:
            command(0xc0);
            _todo = TODO_SEND_TEXT1;
            i = 0;
            _waitAfterMs = 10;
            break;
        case TODO_SEND_TEXT1:
            if (_pText1 && _pText1[i]) write(_pText1[i++]);
            else                       _todo = TODO_READY;
            _waitAfterMs = 5;
            break;
    }
}

void LcdMain()
{
    handleDoIt();
    static uint32_t msTimerTick = 0;
    static char pTime0[16];
    static char pTime1[16];
    if (MsTimerRepetitive(&msTimerTick, 1000) && _todo == TODO_READY)
    {
        struct tm tm;
        ClkNowTmUtc(&tm);
        strftime(pTime0, sizeof(pTime0), "%a, %d %b %Y", &tm);
        strftime(pTime1, sizeof(pTime1), "%H:%M:%S GMT", &tm);
        LcdSendText(pTime0, pTime1);
    }
}