#include "mbed.h"
#include "uart.h"
#include  "esp.h"
#include   "at.h"
#include "wifi.h"
#include  "log.h"
#include   "io.h"
#include  "cfg.h"

#define IP_WAIT_TIME_MS 5000

int WifiStatus;

/*
START
=====
AT_SUCCESS + WIFI_GOT_IP    -> MUX         Under normal circumstances the module will start at the correct baud and already set up for the wifi.
AT_TIMEOUT + WIFI_CONNECTED -> GET_STATUS  Connected to WiFi but no IP given - give the wifi a bit longer
AT_TIMEOUT + WIFI_READY     -> GET_STATUS  Not connected to WiFi - give the wifi a bit longer
AT_TIMEOUT + WIFI_NONE      -> AUTOBAUD    Not even the module ready message was received: probably the baud rate is wrong so check each baud.

AUTOBAUD
========
Changes the baud rate; sends AT and waits for OK
AT_SUCCESS                  -> CHANGE_BAUD Found a baud rate which works so now need to change to the correct baud rate
AT_TIMEOUT                                 Try the next baud rate

CHANGE_BAUD
===========
AT_SUCCESS                  -> GET_STATUS  Don't know the wifi status at this point so find it

GET_STATUS
==========
wait until some time has passed since start
AT_SUCCESS + WIFI_GOT_IP    -> MUX
AT_SUCCESS + WIFI_CONNECTED -> BOMB_OUT    Connected to wifi but could not get an IP
AT_SUCCESS + WIFI_READY     -> CONNECT     This happens when the module has never been, or has lost its connection to, the wifi.
AT_SUCCESS + WIFI_NONE      -> BOMB_OUT    This should not happen

CONNECT
=======
AT_SUCCESS                  -> GET_STATUS

AT_MUX
======

AT_VERSION
==========

AT_STARTED
==========

*/

#define AM_STOP        0
#define AM_START       1
#define AM_AUTOBAUD    2
#define AM_CHANGE_BAUD 3
#define AM_CONNECT     4
#define AM_GET_STATUS  5
#define AM_WAIT_WIFI   6
#define AM_MUX         7
#define AM_VERSION     8
#define AM_STARTED     9
static int am = AM_STOP;
int WifiStarted() { return am == AM_STARTED; }

int WifiMain()
{
    if (AtBusy()) return 0;
    
    static int result = AT_NONE;
    static int autoBaudMultiplier = 1;
    static int autoBaudOdd = false;
    static int autoBaudRate = 0;
    
    static Timer wifiWaitTimer;
    
    switch (am)
    {
        case AM_STOP:
            AtResetAndStop();
            am = AM_START;
            result = AT_NONE;
            break;
        case AM_START:
            switch (result)
            {
                case AT_NONE:
                    AtReleaseResetAndStart(&result);
                    wifiWaitTimer.reset();
                    wifiWaitTimer.start();
                    break;
                case AT_SUCCESS:
                    switch (WifiStatus)
                    {
                        case WIFI_GOT_IP:
                            am = AM_MUX;
                            result = AT_NONE;
                            break;
                        default:
                            LogF("Started WiFi and expected WIFI_GOT_IP but had %d", WifiStatus);
                            return -1;
                    }
                    break;
                default:
                    switch (WifiStatus)
                    {
                        case WIFI_READY:
                            am = AM_CONNECT;
                            result = AT_NONE;
                            break;
                        default:
                            am = AM_AUTOBAUD;
                            result = AT_NONE;
                            autoBaudMultiplier = 1;
                            autoBaudOdd = false;
                            break;
                    }
                    break;
            }
            break;
        case AM_AUTOBAUD:
            switch (result)
            {
                case AT_NONE:
                    autoBaudRate = 9600 * autoBaudMultiplier;
                    if (autoBaudOdd) autoBaudRate += autoBaudRate >> 1; //Multiply by 1.5
                    UartBaud(autoBaudRate);
                    AtAt(&result);
                    break;
                case AT_SUCCESS:
                    LogF("Connected successfully at a baud rate of %d\r\n", autoBaudRate);
                    am = AM_CHANGE_BAUD;
                    result = AT_NONE;
                    break;
                default:
                    if (autoBaudOdd) autoBaudMultiplier <<= 1;
                    autoBaudOdd = !autoBaudOdd;
                    if (autoBaudMultiplier > 64)
                    {
                        LogCrLf("Could not find a baud rate to connect to ESP");
                        return -1;
                    }
                    result = AT_NONE;
                    break;
            }
            break;
        case AM_CHANGE_BAUD:
            switch (result)
            {
                case AT_NONE:
                    LogF("Changing baud rate to %d\r\n", CfgBaud);
                    AtBaud(CfgBaud, &result);
                    break;
                case AT_SUCCESS:
                    UartBaud(CfgBaud);
                    am = AM_GET_STATUS;
                    result = AT_NONE;
                    break;
                default:
                    LogCrLf("Could not change baud");
                    return -1;
            }
            break;
        case AM_CONNECT:
            switch (result)
            {
                case AT_NONE:
                    AtConnect(CfgSsid, CfgPassword, &result);
                    break;
                case AT_SUCCESS:
                    wifiWaitTimer.reset();
                    wifiWaitTimer.start();
                    am = AM_GET_STATUS;
                    result = AT_NONE;
                    break;
                default:
                    LogCrLf("Could not connect to WiFi");
                    return -1;
            }
            break;
        case AM_GET_STATUS:
            if (wifiWaitTimer.read_ms() < IP_WAIT_TIME_MS) break; //Do nothing until enough time has passed
            wifiWaitTimer.stop();
            switch (result)
            {
                case AT_NONE:
                    AtGetEspStatus(&result);
                    break;
                case AT_SUCCESS:
                    switch(AtEspStatus)
                    {
                        case 2:
                            WifiStatus = WIFI_GOT_IP;
                            am = AM_MUX;
                            break;
                        case 3:
                            WifiStatus = WIFI_CONNECTED;
                            LogCrLf("Could connect but not get an IP address");
                            return -1;
                        case 4:
                            WifiStatus = WIFI_READY;
                            am = AM_CONNECT;
                            break;
                        default:
                            LogF("Unknown CIPSTATUS --> STATUS:%d", AtEspStatus);
                            return -1;
                    }
                    result = AT_NONE;
                    break;
                default:
                    LogCrLf("Could not connect to WiFi");
                    return -1;
            }
            break;
        
        case AM_MUX:
            switch (result)
            {
                case AT_NONE:
                    AtMux(&result);
                    break;
                case AT_SUCCESS:
                    am = AM_VERSION;
                    result = AT_NONE;
                    break;
                default:
                    LogCrLf("Could not set up multiple ids");
                    return -1;
            }
            break;
        case AM_VERSION:
            switch (result)
            {
                case AT_NONE:
                    AtGetEspVersion(&result);
                    break;
                case AT_SUCCESS:
                    am = AM_STARTED;
                    result = AT_NONE;
                    break;
                default:
                    LogCrLf("Could not set up multiple ids");
                    return -1;
            }
            break;        
        case AM_STARTED:
            wifiWaitTimer.stop();
            return 0;
        default:
            LogF("Unknown \'am\' %d", am);
            return -1;
    }
   
    return 0;
}
